#include <stdlib.h>
#include <string.h>

#include "area.h"

/*****************************************************************************/
/*                                                                           */
/*       Action Request System External Authentication (AREA) Sample         */
/*                                                                           */
/*****************************************************************************/
/* Description: This is a sample external authentication server.  It is      */
/*    intended for use by Remedy customers in implementing a customized      */
/*    authentication server.                                                 */
/*                                                                           */
/*    This sample allows the user "Good User" to log in and acquire a        */
/*    floating license if one is available.  The user "Bad User" is not      */
/*    allowed to login.  Any other users are unknown and it is up to the     */
/*    AR System server to log the user in as a guest user.                   */
/*                                                                           */
/*****************************************************************************/



/*****************************************************************************/
/*                                                                           */
/*                        AREAInitializationCallback                         */
/*                                                                           */
/*****************************************************************************/
/* Description: This function is called by main() which is defined in the    */
/*    Action Request System External Authentication (AREA) core library.     */
/*                                                                           */
/*    A customized authentication server must define this function.  Any     */
/*    global information or resources should be initialized here.            */
/*                                                                           */
/*    A return value of zero signifies success.  A non-zero return value     */
/*    will cause the process to exit() with the return value as the exit     */
/*    code.                                                                  */
/*                                                                           */
/*****************************************************************************/

int AREAInitializationCallback(
int   argc,
char *argv[]
)
{
   /* This sample has the data allocated and deallocated with every request  */
   /* from the AR System server.  As such, we have nothing to do here.       */

   return 0;
}


/*****************************************************************************/
/*                                                                           */
/*                          AREAVerifyLoginCallback                          */
/*                                                                           */
/*****************************************************************************/
/* Description: This function is called by the AREA core library whenever a  */
/*    request is made by the AR System Server to authenticate a user.        */
/*                                                                           */
/*    The user name and password (unencrypted) are passed as parameters.     */
/*                                                                           */
/*    A customized authentication server must define this function.  Any     */
/*    global information or resources accessed here must be protected with   */
/*    appropriate mutual exclusion locks to be multi-thread safe.            */
/*                                                                           */
/*    A customized authentication server must return a response.  A NULL     */
/*    response will be interpreted as a failed login attempt.                */
/*                                                                           */
/*****************************************************************************/

void AREAVerifyLoginCallback(
ARNameType           user,
ARNameType           password,
ARNameType           networkAddr,
AREAResponseStruct **response
)
{
   *response = (AREAResponseStruct *) malloc (sizeof(AREAResponseStruct));
   if (*response == NULL)
      return;
   
   if (strcmp(user, "Good User") == 0)
   {                                 /* Good User login is successful */
      (*response)->licenseMask  = AREA_LICENSE_MASK_ALL;
      (*response)->licenseWrite = AR_LICENSE_TYPE_FLOATING;
      (*response)->licenseFTS   = AR_LICENSE_TYPE_NONE;
      (*response)->licenseRes1  = AR_LICENSE_TYPE_NONE;
      (*response)->groups       = NULL;
      (*response)->notifyMech   = AR_NOTIFY_VIA_DEFAULT;
      (*response)->email        = NULL;
      (*response)->loginStatus  = AREA_LOGIN_SUCCESS;
      (*response)->messageText  = NULL;
      (*response)->logText      = NULL;
      (*response)->modTime      = 0;
   }
   else if (strcmp(user, "Bad User") == 0)
   {                                 /* Bad User login fails */
      (*response)->loginStatus  = AREA_LOGIN_FAILED;
      (*response)->messageText  = NULL;
      (*response)->logText      = NULL;
   }
   else
   {                                 /* otherwise, we do not know the user */
      (*response)->loginStatus  = AREA_LOGIN_UNKNOWN_USER;
      (*response)->messageText  = NULL;
      (*response)->logText      = NULL;
   }

   return;
}


/*****************************************************************************/
/*                                                                           */
/*                             AREANeedToSyncCallback                        */
/*                                                                           */
/*****************************************************************************/
/* Description: This function is called by the AREA core library             */
/*    periodically at the request of the AR System server.                   */
/*                                                                           */
/*    A customized authentication server must define this function.  Any     */
/*    global information or resources accessed here must be protected with   */
/*    appropriate mutual exclusion locks to be multi-thread safe.            */
/*                                                                           */
/*    A non-zero return value will instruct the AR System server to flush    */
/*    its cache of user information.                                         */
/*                                                                           */
/*****************************************************************************/

int AREANeedToSyncCallback(void)
{
   return 0;
}


/*****************************************************************************/
/*                                                                           */
/*                               AREAFreeCallback                            */
/*                                                                           */
/*****************************************************************************/
/* Description: This function is called by the AREA core library after a     */
/*    response is made to the AR System Server to authenticate a user.       */
/*                                                                           */
/*    A pointer to the response structure returned by                        */
/*    AREAVerifyLoginCallback() is passed as a parameter.                    */
/*                                                                           */
/*    A customized authentication server must define this function.  Any     */
/*    global information or resources accessed here must be protected with   */
/*    appropriate mutual exclusion locks to be multi-thread safe.            */
/*                                                                           */
/*    A customized authentication server must return a response.  A NULL     */
/*    response will be interpreted as a failed login attempt.                */
/*                                                                           */
/*****************************************************************************/

void AREAFreeCallback(
AREAResponseStruct *response
)
{
   /* Since we allocated the response in AREAVerifyLoginCallback(), we */
   /* should free it here.                                             */

   if (response != NULL)
      (void) free (response);

   return;
}


/*****************************************************************************/
/*                                                                           */
/*                           AREATerminationCallback                         */
/*                                                                           */
/*****************************************************************************/
/* Description: This function is called by the AREA core library after a     */
/*    response is made to the AR System Server to authenticate a user.       */
/*                                                                           */
/*    Resources created in AREAInitializationCallback may be destroyed here. */
/*                                                                           */
/*    A customized authentication server must define this function.  Any     */
/*    global information or resources accessed here must be protected with   */
/*    appropriate mutual exclusion locks to be multi-thread safe.            */
/*                                                                           */
/*****************************************************************************/

void AREATerminationCallback(void)
{
   /* We have nothing to do here since we did not allocate or create any     */
   /* global resources in AREAInitializationCallback().                      */

   return;
}
