/* File:  print.c */

#include <stdio.h>
#include <stdarg.h>
#include <string.h>

#include "ar.h"
#include "arstruct.h"
#include "globals.h"

#include "api.h"
#include "print.h"
#include "util.h"

#ifndef OUTPUT_MODE
void PrintAuxillaryStatus()
{
   /* perform any generic printing operations here */
}

#else
void PrintAuxillaryStatus();
#endif

/*****************************************************************************/
/*                                                                           */
/*                             DriverPrintResult                             */
/*                                                                           */
/*****************************************************************************/

void DriverPrintResult(const char *format, ...)
{
   va_list marker;

   if (!(gQuietMode & SUPPRESS_RESULTS))
   {
      va_start(marker, format);

      vfprintf(((ThreadControlBlock *) GetThreadControlBlockPtr())->outFile,
               format, marker);

      va_end(marker);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          DriverPrintResultHeader                          */
/*                                                                           */
/*****************************************************************************/

void DriverPrintResultHeader(char *functionName)
{
   if (!(gQuietMode & SUPPRESS_RESULTS))
      fprintf(((ThreadControlBlock *) GetThreadControlBlockPtr())->outFile,
              "\n   %s  results\n", functionName);
}


/*****************************************************************************/
/*                                                                           */
/*                             DriverPrintHeader                             */
/*                                                                           */
/*****************************************************************************/

void DriverPrintHeader(char *header)
{
   if (!(gQuietMode & SUPPRESS_HEADERS) &&
       ((ThreadControlBlock *) GetThreadControlBlockPtr())->primaryThread)
      printf("%s\n", header);
}


/*****************************************************************************/
/*                                                                           */
/*                            DriverPrintPrompt                              */
/*                                                                           */
/*****************************************************************************/

void DriverPrintPrompt(const char *format, ...)
{
   va_list marker;

   if (!(gQuietMode & SUPPRESS_PROMPTS) &&
       ((ThreadControlBlock *) GetThreadControlBlockPtr())->primaryThread)
   {
      va_start(marker, format);

      vfprintf(stdout, format, marker);

      va_end(marker);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                              DriverPrintMenu                              */
/*                                                                           */
/*****************************************************************************/

void DriverPrintMenu(char *menuLine)
{
   if (!(gQuietMode & SUPPRESS_MENU))
      printf("%s", menuLine);
}


/*****************************************************************************/
/*                                                                           */
/*                             DriverPrintError                              */
/*                                                                           */
/*****************************************************************************/

void DriverPrintError(const char *format, ...)
{
   va_list marker;

   if (!(gQuietMode & SUPPRESS_ERRORS))
   {
      va_start(marker, format);

      vfprintf(((ThreadControlBlock *) GetThreadControlBlockPtr())->outFile,
               format, marker);

      va_end(marker);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            DriverPrintWarning                             */
/*                                                                           */
/*****************************************************************************/

void DriverPrintWarning(const char *format, ...)
{
   va_list marker;

   if (!(gQuietMode & SUPPRESS_WARNINGS))
   {
      va_start(marker, format);

      vfprintf(((ThreadControlBlock *) GetThreadControlBlockPtr())->outFile,
               format, marker);

      va_end(marker);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARBoolean                              */
/*                                                                           */
/*****************************************************************************/

void PrintARBoolean(header, value)
char      *header;            /* IN; header string for the value */
ARBoolean  value;             /* IN; value to print */

{
   DriverPrintResult((value) ? "%s TRUE\n" : "%s FALSE\n", header);
}


/*****************************************************************************/
/*                                                                           */
/*                                  PrintInt                                 */
/*                                                                           */
/*****************************************************************************/

void PrintInt(header, value)
char      *header;            /* IN; header string for the value */
int        value;             /* IN; value to print */

{
   DriverPrintResult("%s %d\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                                  PrintLong                                */
/*                                                                           */
/*****************************************************************************/

void PrintLong(header, value)
char      *header;            /* IN; header string for the value */
long       value;             /* IN; value to print */

{
   DriverPrintResult("%s %ld\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                                  PrintULong                               */
/*                                                                           */
/*****************************************************************************/

void PrintULong(header, value)
char          *header;            /* IN; header string for the value */
unsigned long  value;             /* IN; value to print */

{
   DriverPrintResult("%s %lu\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                                 PrintReal                                 */
/*                                                                           */
/*****************************************************************************/

void PrintReal(header, value)
char      *header;            /* IN; header string for the value */
double     value;             /* IN; value to print */

{
   DriverPrintResult("%s %lf\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                                 PrintChar                                 */
/*                                                                           */
/*****************************************************************************/

void PrintChar(header, value)
char      *header;            /* IN; header string for the value */
char      *value;             /* IN; value to print */

{
   if (value == NULL)
      DriverPrintResult("%s\n", header);
   else
      DriverPrintResult("%s %s\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                             PrintARTimestamp                              */
/*                                                                           */
/*****************************************************************************/

void PrintARTimestamp(header, value)
char       *header;           /* IN; header string for the value */
ARTimestamp value;            /* IN; value to print */

{
   DriverPrintResult("%s %ld\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                           PrintARTimestampList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARTimestampList(header, header2, value)
char             *header;     /* IN; header string for the value */
char             *header2;    /* IN; header string for 2nd+ line */
ARTimestampList  *value;      /* IN; value to print */

{
   char          buff[82];    /* buffer to build output in */
   char          buff2[16];   /* buffer to hold a converted number */
   int           i;           /* working index */
   ARTimestamp  *tempPtr;     /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->timestampList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         (void) sprintf(buff2, "%lu", *tempPtr);
         if ((strlen(buff) + strlen(buff2)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrintResult("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, buff2);
         tempPtr++;
      }
   }
   DriverPrintResult("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARUnsignedIntList                           */
/*                                                                           */
/*****************************************************************************/

void PrintARUnsignedIntList(header, header2, value)
char              *header;    /* IN; header string for the value */
char              *header2;   /* IN; header string for 2nd+ line */
ARUnsignedIntList *value;     /* IN; value to print */

{
   char          buff[82];    /* buffer to build output in */
   char          buff2[16];   /* buffer to hold a converted number */
   int           i;           /* working index */
   unsigned int *tempPtr;     /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->intList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         (void) sprintf(buff2, "%u", *tempPtr);
         if ((strlen(buff) + strlen(buff2)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrintResult("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, buff2);
         tempPtr++;
      }
   }
   DriverPrintResult("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintAREntryIdList                            */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryIdList(header, header2, value)
char          *header;        /* IN; header string for the value */
char          *header2;       /* IN; header string for 2nd+ line */
AREntryIdList *value;         /* IN; value to print */

{
   char           buff[82];   /* buffer to build output in */
   int            i;          /* working index */
   AREntryIdType *tempPtr;    /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->entryIdList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         if ((strlen(buff) + strlen(*tempPtr)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrintResult("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, *tempPtr);
         tempPtr++;
      }
   }
   DriverPrintResult("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARInternalId                           */
/*                                                                           */
/*****************************************************************************/

void PrintARInternalId(header, value)
char        *header;          /* IN; header string for the value */
ARInternalId value;           /* IN; value to print */

{
   DriverPrintResult("%s %lu\n", header, value);
}          


/*****************************************************************************/
/*                                                                           */
/*                           PrintARInternalIdList                           */
/*                                                                           */
/*****************************************************************************/

void PrintARInternalIdList(header, header2, value)
char             *header;     /* IN; header string for the value */
char             *header2;    /* IN; header string for 2nd+ line */
ARInternalIdList *value;      /* IN; value to print */

{
   char          buff[82];    /* buffer to build output in */
   char          buff2[16];   /* buffer to hold a converted number */
   int           i;           /* working index */
   ARInternalId *tempPtr;     /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->internalIdList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         (void) sprintf(buff2, "%lu", *tempPtr);
         if ((strlen(buff) + strlen(buff2)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrintResult("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, buff2);
         tempPtr++;
      }
   }
   DriverPrintResult("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                              PrintARNameList                              */
/*                                                                           */
/*****************************************************************************/

void PrintARNameList(header, header2, value)
char       *header;           /* IN; header string for the value */
char       *header2;          /* IN; header string for 2nd+ line */
ARNameList *value;            /* IN; value to print */

{
   char        buff[82];      /* buffer to build output in */
   int         i;             /* working index */
   ARNameType *tempPtr;       /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->nameList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         if ((strlen(buff) + strlen(*tempPtr)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrintResult("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, *tempPtr);
         tempPtr++;
      }
   }
   DriverPrintResult("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARServerNameList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARServerNameList(header, header2, value)
char             *header;     /* IN; header string for the value */
char             *header2;    /* IN; header string for 2nd+ line */
ARServerNameList *value;      /* IN; value to print */

{
   char              buff[82];/* buffer to build output in */
   int               i;       /* working index */
   ARServerNameType *tempPtr; /* working pointer */

   (void) sprintf(buff, "%s (%u)  ", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->nameList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         if ((strlen(buff) + strlen(*tempPtr)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrintResult("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, *tempPtr);
         tempPtr++;
      }
   }
   DriverPrintResult("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARTextStringList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARTextStringList(header, header2, value)
char             *header;     /* IN; header string for the value */
char             *header2;    /* IN; header string for 2nd+ item */
ARTextStringList *value;      /* IN; value to print */

{
   int               i;       /* working index */

   DriverPrintResult("%s : %u items\n", header, value->numItems);
   for (i = 0; i < (int) value->numItems; i++)
      PrintChar(header2, value->stringList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARControlStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARControlStruct(value)
ARControlStruct *value;       /* IN; value to print */

{
   DriverPrintResult("Control record:\n");
   PrintChar("   User name: ", value->user);
   PrintChar("   Password : ", value->password);
   PrintChar("   Language : ", value->language);
   PrintARTimestamp("   Operation time:", value->operationTime);
   PrintChar("   Server   : ", value->server);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARStatusStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARStatusStruct(value)
ARStatusStruct *value;        /* IN; value to print */

{
   DriverPrintResult("Status Struct :\n");
   DriverPrintResult("   Message type : ");
   switch (value->messageType)
   {
      case AR_RETURN_OK         :
         DriverPrintResult("NOTE\n");
         break;
      case AR_RETURN_WARNING    :
         DriverPrintResult("WARNING\n");
         break;
      case AR_RETURN_ERROR      :
         DriverPrintResult("ERROR\n");
         break;
      default                   :
         DriverPrintResult("<unknown type>\n");
         break;
   }
   PrintLong("   Message number : ", value->messageNum);
   PrintChar("   Message: ", value->messageText);
   if (value->appendedText && value->appendedText[0])
      PrintChar("   Appended: ", value->appendedText);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARStatusList                              */
/*                                                                           */
/*****************************************************************************/

void PrintARStatusList(value)
ARStatusList *value;          /* IN; value to print */

{
   int             i;         /* working index */
   ARStatusStruct *tempPtr;   /* working pointer */

   DriverPrintResult("Status List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->statusList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARStatusStruct(tempPtr);
         tempPtr++;
      }
   }
   PrintAuxillaryStatus();
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARByteList                                */
/*                                                                           */
/*****************************************************************************/

void PrintARByteList(header, header2, value)
char          *header;        /* IN; header string for the value */
char          *header2;       /* IN; header string for 2nd+ line */
ARByteList    *value;         /* IN; value to print */
{
   int             i;
   int             length;
   unsigned char  *tempPtr;

   DriverPrintResult("%s  type - %lu with %u bytes", header, value->type,
                     value->numItems);
   length = (80 - strlen(header2)) / 2;
   if (value->numItems != 0)
   {  /* Print the stream of bytes */
      tempPtr = value->bytes;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if ((i % length) == 0)
            DriverPrintResult("\n%s", header2);
         DriverPrintResult("%02X", *tempPtr);
         tempPtr++;
      }
      DriverPrintResult("\n");
    }
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARCoordList                               */
/*                                                                           */
/*****************************************************************************/

void PrintARCoordList(header, header2, value)
char          *header;        /* IN; header string for the value */
char          *header2;       /* IN; header string for 2nd+ line */
ARCoordList   *value;         /* IN; value to print */
{
   int             i;
   ARCoordStruct  *tempPtr;

   DriverPrintResult("%s %u items\n", header, value->numItems);
   if (value->numItems != 0)
   {
      /* Print the pairs of coordinates */
      tempPtr = value->coords;
      for (i = 0; i < (int) value->numItems; i++)
      {
         DriverPrintResult("%sx: %ld, y: %ld\n", header2, tempPtr->x, tempPtr->y);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                            PrintARAttachLocStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARAttachLocStruct(header, value)
char           *header;        /* IN; header string for the value */
ARLocStruct    *value;         /* IN; value to print */
{
   DriverPrintResult("      %s :\n", header);

   if (value->locType == 0)
      DriverPrintResult("            Locator Type : (NULL)");
   else
   {
      DriverPrintResult("            Locator Type : %ld\n", value->locType);
      if (value->locType == AR_LOC_FILENAME)
         DriverPrintResult("            File Name    : %s\n",
                           value->u.filename);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                            PrintARAttachProps                             */
/*                                                                           */
/*****************************************************************************/

void PrintARAttachProps(header, value)
char              *header;        /* IN; header string for the value */
ARAttachStruct    *value;         /* IN; value to print */
{
   DriverPrintResult("%s\n", header);

   if (value->name[0] != '\0')
      DriverPrintResult("      Name            : %s\n", value->name);
   else
      DriverPrintResult("      Name            : <null>\n");

   DriverPrintResult("      Original size   : %ld\n", value->origSize);
   DriverPrintResult("      Compressed size : %ld\n", value->compSize);
   PrintARAttachLocStruct("Locator", &value->loc);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARValueStruct                             */
/*                                                                           */
/*****************************************************************************/

void PrintARValueStruct(header, value)
char          *header;        /* IN; header string for the value */
ARValueStruct *value;         /* IN; value to print */

{
   char  buff[82];         /* buffer to hold second line header */

   DriverPrintResult("%sValue:  ", header);
   switch (value->dataType)
   {
      case AR_DATA_TYPE_NULL    :  /* NULL value (incl. no default) */
         DriverPrintResult("NULL\n");
         break;
      case AR_DATA_TYPE_KEYWORD :
         PrintInt("(keyword)  ", value->u.keyNum);
         break;
      case AR_DATA_TYPE_INTEGER :
         PrintLong("(integer)  ", (int) value->u.intVal);
         break;
      case AR_DATA_TYPE_REAL    :
         PrintReal("(real)  ", value->u.realVal);
         break;
      case AR_DATA_TYPE_CHAR    :
         PrintChar("(char)  ", value->u.charVal);
         break;
      case AR_DATA_TYPE_DIARY   :
         APIPrintDiary("(diary)  ", value->u.diaryVal);
         break;
      case AR_DATA_TYPE_ENUM    :
         PrintLong("(enumerated)  ", (int) value->u.enumVal);
         break;
      case AR_DATA_TYPE_TIME    :
         PrintARTimestamp("(timestamp)  ", value->u.timeVal);
         break;
      case AR_DATA_TYPE_BITMASK :
         PrintLong("(bitmask)  ", (int) value->u.maskVal);
         break;
      case AR_DATA_TYPE_BYTES :
         (void) sprintf(buff, "%s        ", header);
         PrintARByteList("(bytelist)  ", buff, value->u.byteListVal);
         break;
      case AR_DATA_TYPE_JOIN :
         DriverPrintResult("(join)\n");
         break;
      case AR_DATA_TYPE_TRIM :
         DriverPrintResult("(trim)\n");
         break;
      case AR_DATA_TYPE_CONTROL:
         DriverPrintResult("(control)\n");
         break;
      case AR_DATA_TYPE_ULONG :
         PrintULong("(ulong)  ", (int) value->u.ulongVal);
         break;
      case AR_DATA_TYPE_COORDS:
         (void) sprintf(buff, "%s        ", header);
         PrintARCoordList("(coordlist)  ", buff, value->u.coordListVal);
         break;
      case AR_DATA_TYPE_PAGE:
         DriverPrintResult("(Page)\n");
         break;
      case AR_DATA_TYPE_PAGE_HOLDER:
         DriverPrintResult("(PageHolder)\n");
         break;
      case AR_DATA_TYPE_ATTACH:
         PrintARAttachProps("(attachment)  ", value->u.attachVal);
         break;
      case AR_DATA_TYPE_DECIMAL :
         PrintChar("(decimal)  ", value->u.decimalVal);
         break;
      default                   :
         DriverPrintResult("<undefined data type tag = %u>\n",
                           value->dataType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARValueList                              */
/*                                                                           */
/*****************************************************************************/

void PrintARValueList(header, value)
char        *header;          /* IN; header string for the value */
ARValueList *value;           /* IN; value to print */

{
   char            buff[45];  /* buffer to build nested header in */
   int             i;         /* working index */
   ARValueStruct  *tempPtr;   /* working pointer */

   DriverPrintResult("%sValue List : %u items\n", header,value->numItems);
   if (value->numItems != 0)
   {
      if (strlen(header) >= (size_t) 40)
         buff[0] = '\0';
      else
         (void) sprintf(buff, "%s   ", header);

      tempPtr = value->valueList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARValueStruct(buff, tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARValueListList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARValueListList(value)
ARValueListList *value;       /* IN; value to print */

{
   int             i;         /* working index */
   ARValueList    *tempPtr;   /* working pointer */

   DriverPrintResult("Value List List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->valueListList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARValueList("   ", tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintAREntryListFieldStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListFieldStruct(value)
AREntryListFieldStruct *value;     /* IN; value to print */

{
   DriverPrintResult("   Entry List Field Struct:\n");
   PrintLong("      Field Id :", value->fieldId);
   PrintInt("      Column Width:", value->columnWidth);
   PrintChar("      Separator:", value->separator);
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintAREntryListFieldList                          */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListFieldList(value)
AREntryListFieldList *value;  /* IN; value to print */

{
   int                     i;      /* working index */
   AREntryListFieldStruct *tempPtr;/* working pointer */

   DriverPrintResult("Entry List Field List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->fieldsList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintAREntryListFieldStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARSortList                               */
/*                                                                           */
/*****************************************************************************/

void PrintARSortList(value)
ARSortList *value;      /* IN; value to print */

{
   int            i;      /* working index */

   DriverPrintResult("Sort List : %u items\n", value->numItems);
   for (i = 0; i < (int) value->numItems; i++)
   {
      DriverPrintResult("   Sort Struct:\n");
      PrintLong("      Field Id :", value->sortList[i].fieldId);
      switch (value->sortList[i].sortOrder)
      {
         case AR_SORT_ASCENDING  :
            DriverPrintResult("      Sort Order: ASCENDING\n");
            break;
         case AR_SORT_DESCENDING :
            DriverPrintResult("      Sort Order: DESCENDING\n");
            break;
         default              :
            DriverPrintResult("      Sort Order: <undefined tag value = %u>\n",
                              value->sortList[i].sortOrder);
            break;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintAREntryListStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListStruct(value)
AREntryListStruct *value;     /* IN; value to print */

{
   unsigned int i;

   DriverPrintResult("Entry List Struct:\n");
   if (value != NULL && value->entryId.numItems > 0)
   {
      DriverPrintResult("   Entry Id : %s", value->entryId.entryIdList[0]);
      for (i = 1; i < value->entryId.numItems; i++)
         DriverPrintResult("|%s", value->entryId.entryIdList[i]);
      DriverPrintResult("\n");
      PrintChar("   Short Description :", value->shortDesc);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintAREntryListList                            */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListList(value)
AREntryListList *value;       /* IN; value to print */

{
   int                i;      /* working index */
   AREntryListStruct *tempPtr;/* working pointer */

   DriverPrintResult("Entry List List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->entryList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintAREntryListStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFieldValueStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldValueStruct(value)
ARFieldValueStruct *value;    /* IN; value to print */

{
   DriverPrintResult("Field Value Struct:\n");
   PrintARInternalId("   Field Id :", value->fieldId);
   if ((value->fieldId == AR_CORE_STATUS_HISTORY) &&
       (value->value.dataType == AR_DATA_TYPE_CHAR))
      APIPrintStatusHistory("   Value: ", value->value.u.charVal);
   else
      PrintARValueStruct("   ", &value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFieldValueList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldValueList(value)
ARFieldValueList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARFieldValueStruct *tempPtr;/* working pointer */

   DriverPrintResult("Field Value List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->fieldValueList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFieldValueStruct(tempPtr);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                      PrintARFieldValueListList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldValueListList(value)
ARFieldValueListList *value;      /* IN; value to print */

{
   int               i;       /* working index */
   ARFieldValueList *tempPtr; /* working pointer */

   DriverPrintResult("Field Value List List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->valueListList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFieldValueList(tempPtr);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintAREntryListFieldValueStruct                */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListFieldValueStruct(value)
AREntryListFieldValueStruct *value;     /* IN; value to print */

{
   unsigned int i;

   DriverPrintResult("Entry List Struct:\n");
   if (value != NULL && value->entryId.numItems > 0)
   {
      DriverPrintResult("   Entry Id : %s", value->entryId.entryIdList[0]);
      for (i = 1; i < value->entryId.numItems; i++)
         DriverPrintResult("|%s", value->entryId.entryIdList[i]);
      DriverPrintResult("\n");
      DriverPrintResult("Entry Field/Value List:\n");
      PrintARFieldValueList(value->entryValues);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintAREntryListFieldValueList                  */
/*                                                                           */
/*****************************************************************************/

void PrintAREntryListFieldValueList(value)
AREntryListFieldValueList *value;       /* IN; value to print */

{
   int                          i;      /* working index */
   AREntryListFieldValueStruct *tempPtr;/* working pointer */

   DriverPrintResult("Entry List List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->entryList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintAREntryListFieldValueStruct(tempPtr);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                      PrintARFieldValueOrArithStruct                       */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldValueOrArithStruct(header, value)
char                      *header;/* IN; header string for the value */
ARFieldValueOrArithStruct *value; /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */
   char   buff2[65];          /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%sField, Value, or Arithmetic Struct:\n", header);
   switch (value->tag)
   {
      case AR_FIELD        :
         DriverPrintResult("%s   Tag: FIELD\n", header);
         (void) sprintf(buff2, "%s   Field Id :", header);
         PrintARInternalId(buff2, value->u.fieldId);
         break;
      case AR_VALUE        :
         DriverPrintResult("%s   Tag: VALUE\n", header);
         PrintARValueStruct(buff, &value->u.value);
         break;
      case AR_ARITHMETIC   :
         DriverPrintResult("%s   Tag: ARITHMETIC\n", header);
         PrintARArithOpStruct(buff, (ARArithOpStruct *) value->u.arithOp);
         break;
      case AR_STAT_HISTORY :
         DriverPrintResult("%s   Tag: STAT HISTORY\n", header);
         (void) sprintf(buff2, "%s   Enumerated value index :", header);
         PrintInt(buff2, (int) value->u.statHistory.enumVal);
         switch (value->u.statHistory.userOrTime)
         {
            case AR_STAT_HISTORY_USER :
               DriverPrintResult("%s   User or Time: USER\n", header);
               break;
            case AR_STAT_HISTORY_TIME :
               DriverPrintResult("%s   User or Time: TIME\n", header);
               break;
            default                   :
               DriverPrintResult("%s   User or Time: %s%u>\n", header,
                                 "<undefined value = ",
                                 value->u.statHistory.userOrTime);
               break;
         }
         break;
      case AR_VALUE_SET    :
         DriverPrintResult("%s   Tag: VALUE SET\n", header);
         PrintARValueList(buff, &value->u.valueSet);
         break;
      case AR_FIELD_TRAN   :
         DriverPrintResult("%s   Tag: FIELD TRAN\n", header);
         (void) sprintf(buff2, "%s   Field Id :", header);
         PrintARInternalId(buff2, value->u.fieldId);
         break;
      case AR_FIELD_DB     :
         DriverPrintResult("%s   Tag: FIELD DB\n", header);
         (void) sprintf(buff2, "%s   Field Id :", header);
         PrintARInternalId(buff2, value->u.fieldId);
         break;
      case AR_LOCAL_VARIABLE :
         DriverPrintResult("%s   Tag: LOCAL VARIABLE\n", header);
         (void) sprintf(buff2, "%s   Local Variable number:", header);
         PrintInt(buff2, (int) value->u.variable);
         break;
      case AR_QUERY        :
         DriverPrintResult("%s   Tag: QUERY\n", header);
         (void) sprintf(buff2, "%s   Schema :", header);
         PrintChar(buff2, value->u.queryValue->schema);
         (void) sprintf(buff2, "%s   Server :", header);
         PrintChar(buff2, value->u.queryValue->server);
         PrintARQualifierStruct(buff, value->u.queryValue->qualifier);
         PrintLong(buff2, (int) value->u.queryValue->valueField);
         DriverPrintResult("%s   Multi Value Code:", header);
         switch (value->u.queryValue->multiMatchCode)
         {
            case AR_QUERY_VALUE_MULTI_ERROR :
               DriverPrintResult("ERROR\n");
               break;
            case AR_QUERY_VALUE_MULTI_FIRST :
               DriverPrintResult("FIRST\n");
               break;
            case AR_QUERY_VALUE_MULTI_SET   :
               DriverPrintResult("SET\n");
               break;
            default                         :
               DriverPrintResult("<undefined multi-match value = %u>\n",
                                 value->u.queryValue->multiMatchCode);
               break;
         }
         break;
      case AR_FIELD_CURRENT:
         DriverPrintResult("%s   Tag: FIELD (current screen)\n", header);
         (void) sprintf(buff2, "%s   Field Id :", header);
         PrintARInternalId(buff2, value->u.fieldId);
         break;
      default              :
         DriverPrintResult("%s   Tag: <undefined tag value = %u>\n", header,
                           value->tag);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARArithOpStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARArithOpStruct(header, value)
char            *header;      /* IN; header string for the value */
ARArithOpStruct *value;       /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%sArithmetic Operation:\n", header);
   switch (value->operation)
   {
      case AR_ARITH_OP_ADD      :
         DriverPrintResult("%s   Operation: ADD\n", header);
         break;
      case AR_ARITH_OP_SUBTRACT :
         DriverPrintResult("%s   Operation: SUBTRACT\n", header);
         break;
      case AR_ARITH_OP_MULTIPLY :
         DriverPrintResult("%s   Operation: MULTIPLY\n", header);
         break;
      case AR_ARITH_OP_DIVIDE   :
         DriverPrintResult("%s   Operation: DIVIDE\n", header);
         break;
      case AR_ARITH_OP_MODULO   :
         DriverPrintResult("%s   Operation: MODULO\n", header);
         break;
      case AR_ARITH_OP_NEGATE   :
         DriverPrintResult("%s   Operation: NEGATE (unary minus)\n", header);
         break;
      default                   :
         DriverPrintResult("%s   Operation: <undefined operation = %u>\n",
                           header, value->operation);
         break;
   }
   if (value->operation != AR_ARITH_OP_NEGATE)
      PrintARFieldValueOrArithStruct(buff, &value->operandLeft);
   PrintARFieldValueOrArithStruct(buff, &value->operandRight);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARRelOpStruct                             */
/*                                                                           */
/*****************************************************************************/

void PrintARRelOpStruct(header, value)
char          *header;        /* IN; header string for the value */
ARRelOpStruct *value;         /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%sRelational Operation:\n", header);
   switch (value->operation)
   {
      case AR_REL_OP_EQUAL         :
         DriverPrintResult("%s   Operation: EQUAL\n", header);
         break;
      case AR_REL_OP_GREATER       :
         DriverPrintResult("%s   Operation: GREATER\n", header);
         break;
      case AR_REL_OP_GREATER_EQUAL :
         DriverPrintResult("%s   Operation: GREATER_EQUAL\n", header);
         break;
      case AR_REL_OP_LESS          :
         DriverPrintResult("%s   Operation: LESS\n", header);
         break;
      case AR_REL_OP_LESS_EQUAL    :
         DriverPrintResult("%s   Operation: LESS_EQUAL\n", header);
         break;
      case AR_REL_OP_NOT_EQUAL     :
         DriverPrintResult("%s   Operation: NOT_EQUAL\n", header);
         break;
      case AR_REL_OP_LIKE          :
         DriverPrintResult("%s   Operation: LIKE\n", header);
         break;
      case AR_REL_OP_IN            :
         DriverPrintResult("%s   Operation: IN\n", header);
         break;
      default                      :
         DriverPrintResult("%s   Operation: <undefined operation = %u>\n",
                           header, value->operation);
         break;
   }
   PrintARFieldValueOrArithStruct(buff, &value->operandLeft);
   PrintARFieldValueOrArithStruct(buff, &value->operandRight);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARQualifierStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARQualifierStruct(header, value)
char              *header;    /* IN; header string for the value */
ARQualifierStruct *value;     /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%sQualifier Struct:\n", header);
   switch (value->operation)
   {
      case AR_COND_OP_NONE   :
         /* is a dummy with no value */
         DriverPrintResult("%s   Operation: <none>\n", header);
         break;
      case AR_COND_OP_AND    :
         DriverPrintResult("%s   Operation: AND\n", header);
         PrintARQualifierStruct(buff, value->u.andor.operandLeft);
         PrintARQualifierStruct(buff, value->u.andor.operandRight);
         break;
      case AR_COND_OP_OR     :
         DriverPrintResult("%s   Operation: OR\n", header);
         PrintARQualifierStruct(buff, value->u.andor.operandLeft);
         PrintARQualifierStruct(buff, value->u.andor.operandRight);
         break;
      case AR_COND_OP_NOT    :
         DriverPrintResult("%s   Operation: NOT\n", header);
         PrintARQualifierStruct(buff, value->u.not);
         break;
      case AR_COND_OP_REL_OP :
         DriverPrintResult("%s   Operation: RELATIONAL OPERATION\n", header);
         PrintARRelOpStruct(buff, value->u.relOp);
         break;
      default             :
         DriverPrintResult("%s   Operation: <undefined operation = %u>\n",
                           header, value->operation);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARIndexStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARIndexStruct(value)
ARIndexStruct   *value;       /* IN; value to print */

{
   int   i;                   /* working index */

   DriverPrintResult("Index Struct:\n");
   PrintInt("   Number of Index Fields:", (int) value->numFields);
   for (i = 0; i < (int)value->numFields; i++)
      PrintLong("   Field Id:", value->fieldIds[i]);
   PrintARBoolean("   Unique flag:", value->unique);
}


/*****************************************************************************/
/*                                                                           */
/*                              PrintARIndexList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARIndexList(value)
ARIndexList   *value;         /* IN; value to print */

{
   int              i;        /* working index */
   ARIndexStruct   *tempPtr;  /* working pointer */

   DriverPrintResult("Index List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->indexList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARIndexStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARCompoundSchema                             */
/*                                                                           */
/*****************************************************************************/

void PrintARCompoundSchema(value)
ARCompoundSchema   *value;         /* IN; value to print */

{
   DriverPrintResult("Compound Schema Info:  ");
   switch (value->schemaType)
   {
      case AR_SCHEMA_NONE :
         DriverPrintResult("<none>\n");
         break;
      case AR_SCHEMA_REGULAR :
         DriverPrintResult("REGULAR\n");
         break;
      case AR_SCHEMA_DIALOG :
         DriverPrintResult("DIALOG\n");
         break;
      case AR_SCHEMA_JOIN :
         DriverPrintResult("JOIN\n");
         DriverPrintResult("  Join member A %s \n", value->u.join.memberA);
         DriverPrintResult("  Join member B %s \n", value->u.join.memberB);
         PrintARQualifierStruct("  Join qualification",
                                &value->u.join.joinQual);
         DriverPrintResult("  Join option %d \n", value->u.join.option);
         break;
      case AR_SCHEMA_VIEW :
         DriverPrintResult("VIEW\n");
         DriverPrintResult("  View table name %s \n",
                           value->u.view.tableName);
         DriverPrintResult("  View qualification %s \n",
                           value->u.view.viewQual);
         break;
      default             :
         DriverPrintResult("  <undefined schemaType = %u>\n",
                           value->schemaType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARStatisticsResultStruct                      */
/*                                                                           */
/*****************************************************************************/

void PrintARStatisticsResultStruct(header, value)
char                     *header;  /* IN; header string for the value */
ARStatisticsResultStruct *value;   /* IN; value to print */

{
   char   buff[55];           /* buffer to build nested header in */
   int    i;                  /* working index */

   if (value->groupByValues.numItems != 0)
   {
      if (strlen(header) >= (size_t) 40)
         (void) strcpy(buff, "Grouped by ");
      else
         (void) sprintf(buff, "%sGrouped by ", header);

      for (i = 0; i < (int) value->groupByValues.numItems; i++)
      {
         PrintARValueStruct(buff, &value->groupByValues.valueList[i]);

         if (i == 0)
            if (strlen(header) >= (size_t) 40)
               (void) strcpy(buff, " then by ");
            else
               (void) sprintf(buff, "%s then by ", header);
      }
   }

   if (strlen(header) >= (size_t) 40)
      (void) strcpy(buff, "   Stat ");
   else
      (void) sprintf(buff, "%s   Stat ", header);
   PrintARValueStruct(buff, &value->result);
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARStatisticsResultList                        */
/*                                                                           */
/*****************************************************************************/

void PrintARStatisticsResultList(header, value)
char                   *header;  /* IN; header string for the value */
ARStatisticsResultList *value;   /* IN; value to print */

{
   char                      buff[45];  /* buffer to build nested header in */
   int                       i;         /* working index */
   ARStatisticsResultStruct *tempPtr;   /* working pointer */

   DriverPrintResult("%sStatistics Result List : %u items\n", header,
                     value->numItems);
   if (value->numItems != 0)
   {
      if (strlen(header) >= (size_t) 40)
         buff[0] = '\0';
      else
         (void) sprintf(buff, "%s   ", header);

      tempPtr = value->resultList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARStatisticsResultStruct(buff, tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARObjectPropList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARObjectPropList(header, prefix, value)
char       *header;      /* IN; header string for the value */
char       *prefix;      /* IN; prefix to Prop list on first line */
ARPropList *value;       /* IN; value to print */

{
   char          buff[45];    /* buffer to build nested header in */
   int           i;           /* working index */
   ARPropStruct *tempPtr;     /* working pointer */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%s%sProp List : %u items\n", header, prefix,
                     value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->props;
      for (i = 0; i < (int) value->numItems; i++)
      {
         DriverPrintResult("%sProp Struct:  ", header);
         switch (tempPtr->prop)
         {
            case AR_OPROP_VENDOR_NAME     :
               DriverPrintResult("VENDOR NAME\n");
               break;
            case AR_OPROP_VENDOR_PRODUCT  :
               DriverPrintResult("VENDOR PRODUCT\n");
               break;
            case AR_OPROP_VENDOR_VERSION     :
               DriverPrintResult("VENDOR VERSION\n");
               break;
            case AR_OPROP_GUID               :
               DriverPrintResult("GUID\n");
               break;
            case AR_OPROP_COPYRIGHT          :
               DriverPrintResult("COPYRIGHT\n");
               break;
            case AR_OPROP_SCC_LOCKED_BY      :
               DriverPrintResult("SCC LOCKED BY\n");
               break;
            case AR_OPROP_SCC_VERSION        :
               DriverPrintResult("SCC VERSION\n");
               break;
            case AR_OPROP_SCC_TIMESTAMP      :
               DriverPrintResult("SCC TIMESTAMP\n");
               break;
            case AR_OPROP_SCC_USER           :
               DriverPrintResult("SCC USER\n");
               break;
            case AR_OPROP_SCC_LOCATION       :
               DriverPrintResult("SCC LOCATION\n");
               break;
            case AR_OPROP_SCC_DATA_LOCKED_BY :
               DriverPrintResult("SCC DATA LOCKED BY\n");
               break;
            case AR_OPROP_SCC_DATA_VERSION   :
               DriverPrintResult("SCC DATA VERSION\n");
               break;
            case AR_OPROP_SCC_DATA_TIMESTAMP :
               DriverPrintResult("SCC DATA TIMESTAMP\n");
               break;
            case AR_OPROP_SCC_DATA_USER      :
               DriverPrintResult("SCC DATA USER\n");
               break;
            case AR_OPROP_SCC_DATA_LOCATION  :
               DriverPrintResult("SCC DATA LOCATION\n");
               break;
            default                       :
               DriverPrintResult("   <undefined property = %u>\n", tempPtr->prop);
               break;
         }
         PrintARValueStruct(buff, &tempPtr->value);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARDisplayPropList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARDisplayPropList(header, prefix, value)
char       *header;      /* IN; header string for the value */
char       *prefix;      /* IN; prefix to Prop list on first line */
ARPropList *value;       /* IN; value to print */

{
   char          buff[45];    /* buffer to build nested header in */
   int           i;           /* working index */
   ARPropStruct *tempPtr;     /* working pointer */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%s%sProp List : %u items\n", header, prefix, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->props;
      for (i = 0; i < (int) value->numItems; i++)
      {
         DriverPrintResult("%sProp Struct:  ", header);
         switch (tempPtr->prop)
         {
            case AR_DPROP_TRIM_TYPE       :
               DriverPrintResult("TRIM TYPE\n");
               break;
            case AR_DPROP_CNTL_TYPE       :
               DriverPrintResult("CNTL TYPE\n");
               break;
            case AR_DPROP_BBOX            :
               DriverPrintResult("BBOX\n");
               break;
            case AR_DPROP_VISIBLE         :
               DriverPrintResult("VISIBLE\n");
               break;
            case AR_DPROP_ENABLE          :
               DriverPrintResult("ENABLE\n");
               break;
            case AR_DPROP_HELP            :
               DriverPrintResult("HELP\n");
               break;
            case AR_DPROP_Z_ORDER         :
               DriverPrintResult("Z ORDER\n");
               break;
            case AR_DPROP_COLOR_FILL      :
               DriverPrintResult("COLOR FILL\n");
               break;
            case AR_DPROP_DEPTH_EFFECT    :
               DriverPrintResult("DEPTH EFFECT\n");
               break;
            case AR_DPROP_DEPTH_AMOUNT    :
               DriverPrintResult("DEPTH AMOUNT\n");
               break;
            case AR_DPROP_COLOR_LINE      :
               DriverPrintResult("COLOR LINE\n");
               break;
            case AR_DPROP_COLOR_TEXT      :
               DriverPrintResult("COLOR TEXT\n");
               break;
            case AR_DPROP_PROMPT          :
               DriverPrintResult("PROMPT\n");
               break;
            case AR_DPROP_LABEL           :
               DriverPrintResult("LABEL\n");
               break;
            case AR_DPROP_LABEL_BBOX      :
               DriverPrintResult("LABEL BBOX\n");
               break;
            case AR_DPROP_LABEL_FONT_STYLE:
               DriverPrintResult("LABEL FONT STYLE\n");
               break;
            case AR_DPROP_LABEL_FONT_SIZE :
               DriverPrintResult("LABEL FONT SIZE\n");
               break;
            case AR_DPROP_LABEL_COLOR_TEXT:
               DriverPrintResult("LABEL COLOR TEXT\n");
               break;
            case AR_DPROP_LABEL_JUSTIFY   :
               DriverPrintResult("LABEL JUSTIFY\n");
               break;
            case AR_DPROP_LABEL_ALIGN     :
               DriverPrintResult("LABEL ALIGN\n");
               break;
            case AR_DPROP_LABEL_POS_SECTOR :
               DriverPrintResult("LABEL POS SECTOR\n");
               break;
            case AR_DPROP_LABEL_POS_JUSTIFY:
               DriverPrintResult("LABEL POS JUSTIFY\n");
               break;
            case AR_DPROP_LABEL_POS_ALIGN :
               DriverPrintResult("LABEL POS ALIGN\n");
               break;
            case AR_DPROP_LABEL_COLOR_FILL:
               DriverPrintResult("LABEL COLOR FILL\n");
               break;
            case AR_DPROP_LABEL_COLOR_LINE:
               DriverPrintResult("LABEL COLOR LINE\n");
               break;
            case AR_DPROP_COORDS          :
               DriverPrintResult("COORDS\n");
               break;
            case AR_DPROP_LINE_WIDTH      :
               DriverPrintResult("LINE WIDTH\n");
               break;
            case AR_DPROP_LINE_PATTERN    :
               DriverPrintResult("LINE PATTERN\n");
               break;
            case AR_DPROP_JOINT_STYLE     :
               DriverPrintResult("JOINT STYLE\n");
               break;
            case AR_DPROP_ENDCAP_START    :
               DriverPrintResult("ENDCAP START\n");
               break;
            case AR_DPROP_ENDCAP_END      :
               DriverPrintResult("ENDCAP END\n");
               break;
            case AR_DPROP_DATA_ROWS       :
               DriverPrintResult("DATA ROWS\n");
               break;
            case AR_DPROP_DATA_COLS       :
               DriverPrintResult("DATA COLS\n");
               break;
            case AR_DPROP_DATA_SPIN       :
               DriverPrintResult("DATA SPIN\n");
               break;
            case AR_DPROP_DATA_MENU       :
               DriverPrintResult("DATA MENU\n");
               break;
            case AR_DPROP_DATA_RADIO      :
               DriverPrintResult("DATA RADIO\n");
               break;
            case AR_DPROP_DATA_MENU_BBOX  :
               DriverPrintResult("MENU BBOX\n");
               break;
            case AR_DPROP_DATA_EXPAND_BBOX:
               DriverPrintResult("EXPAND BBOX\n");
               break;
            case AR_DPROP_TEXT            :
               DriverPrintResult("TEXT\n");
               break;
            case AR_DPROP_TEXT_FONT_STYLE :
               DriverPrintResult("TEXT FONT STYLE\n");
               break;
            case AR_DPROP_TEXT_FONT_SIZE  :
               DriverPrintResult("TEXT FONT SIZE\n");
               break;
            case AR_DPROP_JUSTIFY         :
               DriverPrintResult("JUSTIFY\n");
               break;
            case AR_DPROP_ALIGN           :
               DriverPrintResult("ALIGN\n");
               break;
            case AR_DPROP_IMAGE           :
               DriverPrintResult("IMAGE\n");
               break;
            case AR_DPROP_PUSH_BUTTON_IMAGE:
               DriverPrintResult("PUSH BUTTON IMAGE\n");
               break;
            case AR_DPROP_BUTTON_TEXT     :
               DriverPrintResult("BUTTON TEXT\n");
               break;
            case AR_DPROP_BUTTON_2D       :
               DriverPrintResult("BUTTON 2D\n");
               break;
            case AR_DPROP_BUTTON_IMAGE_POSITION :
               DriverPrintResult("BUTTON IMAGE POSITION\n");
               break;
            case AR_DPROP_BUTTON_SCALE_IMAGE :
               DriverPrintResult("BUTTON SCALE IMAGE\n");
               break;
            case AR_DPROP_BUTTON_MAINTAIN_RATIO :
               DriverPrintResult("BUTTON MAINTAIN RATIO\n");
               break;
            case AR_DPROP_MENU_TEXT       :
               DriverPrintResult("MENU TEXT\n");
               break;
            case AR_DPROP_MENU_POS        :
               DriverPrintResult("MENU POS\n");
               break;
            case AR_DPROP_MENU_MODE       :
               DriverPrintResult("MENU MODE\n");
               break;
            case AR_DPROP_MENU_PARENT     :
               DriverPrintResult("MENU PARENT\n");
               break;
            case AR_DPROP_MENU_HELP       :
               DriverPrintResult("MENU HELP\n");
               break;
            case AR_DPROP_TOOLTIP         :
               DriverPrintResult("TOOLTIP\n");
               break;
            case AR_DPROP_TOOLBAR_POS     :
               DriverPrintResult("TOOLBAR POS\n");
               break;
            case AR_DPROP_TOOLBAR_MODE    :
               DriverPrintResult("TOOLBAR MODE\n");
               break;
            case AR_DPROP_TOOLBAR_TEXT    :
               DriverPrintResult("TOOLBAR TEXT\n");
               break;
            case AR_DPROP_TAB_MODE        :
               DriverPrintResult("TAB MODE\n");
               break;
            case AR_DPROP_TAB_COORD       :
               DriverPrintResult("TAB COORD\n");
               break;
            case AR_DPROP_TAB_TEXT        :
               DriverPrintResult("TAB TEXT\n");
               break;
            case AR_DPROP_TAB_ORDER       :
               DriverPrintResult("TAB ORDER\n");
               break;
            case AR_DPROP_DATETIME_POPUP  :
               DriverPrintResult("DATETIME POPUP\n");
               break;
            case AR_DPROP_BACKGROUND_MODE :
               DriverPrintResult("BACKGROUND MODE\n");
               break;
            case AR_DPROP_TAB_NEXT        :
               DriverPrintResult("TAB NEXT\n");
               break;
            case AR_DPROP_DATA_BBOX       :
               DriverPrintResult("DATA BBOX\n");
               break;
            case AR_DPROP_VIEW_GRID_BBOX  :
               DriverPrintResult("VIEW GRID BBOX\n");
               break;
            case AR_DPROP_VUI_DEFAULT     :
               DriverPrintResult("VUI DEFAULT\n");
               break;
            case AR_DPROP_PANE_LAYOUT     :
               DriverPrintResult("PANE LAYOUT\n");
               break;
            case AR_DPROP_DETAIL_PANE_VISIBILITY :
               DriverPrintResult("DETAIL PANE VISIBILITY\n");
               break;
            case AR_DPROP_PROMPT_PANE_VISIBILITY :
               DriverPrintResult("PROMPT PANE VISIBILITY\n");
               break;
            case AR_DPROP_RESULT_PANE_VISIBILITY :
               DriverPrintResult("RESULT PANE VISIBILITY\n");
               break;
            case AR_DPROP_DETAIL_PANE_COLOR :
               DriverPrintResult("DETAIL PANE COLOR\n");
               break;
            case AR_DPROP_DETAIL_PANE_IMAGE :
               DriverPrintResult("DETAIL PANE IMAGE\n");
               break;
            case AR_DPROP_IMAGE_ALIGN     :
               DriverPrintResult("IMAGE ALIGN\n");
               break;
            case AR_DPROP_IMAGE_JUSTIFY   :
               DriverPrintResult("IMAGE JUSTIFY\n");
               break;
            case AR_DPROP_DISPLAY_PARENT  :
               DriverPrintResult("DISPLAY PARENT\n");
               break;
            case AR_DPROP_PAGE_ORDER      :
               DriverPrintResult("PAGE ORDER\n");
               break;
            case AR_DPROP_PAGE_LABEL_DISPLAY :
               DriverPrintResult("PAGE LABEL DISPLAY\n");
               break;
            case AR_DPROP_PAGE_ARRANGEMENT:
               DriverPrintResult("PAGE ARRANGEMENT\n");
               break;
            case AR_DPROP_DEFAULT_PAGE    :
               DriverPrintResult("DEFAULT PAGE\n");
               break;
            case AR_DPROP_TITLE_BAR_ICON_IMAGE :
               DriverPrintResult("TITLE BAR ICON IMAGE\n");
               break;
            case AR_DPROP_DETAIL_PANE_WIDTH :
               DriverPrintResult("DETAIL PANE WIDTH\n");
               break;
            case AR_DPROP_DETAIL_PANE_HEIGHT :
               DriverPrintResult("DETAIL PANE HEIGHT\n");
               break;
            case AR_DPROP_DETAIL_BANNER_VISIBILITY :
               DriverPrintResult("DETAIL BANNER VISIBILITY\n");
               break;
            case AR_DPROP_PROMPT_BANNER_VISIBILITY :
               DriverPrintResult("PROMPT BANNER VISIBILITY\n");
               break;
            case AR_DPROP_RESULT_BANNER_VISIBILITY :
               DriverPrintResult("RESULT BANNER VISIBILITY\n");
               break;
            case AR_DPROP_ALIAS_SINGULAR  :
               DriverPrintResult("ALIAS SINGULAR \n");
               break;
            case AR_DPROP_ALIAS_PLURAL    :
               DriverPrintResult("ALIAS PLURAL\n");
               break;
            case AR_DPROP_ALIAS_SHORT_SINGULAR :
               DriverPrintResult("ALIAS SHORT SINGULAR \n");
               break;
            case AR_DPROP_ALIAS_SHORT_PLURAL :
               DriverPrintResult("ALIAS SHORT PLURAL\n");
               break;
            case AR_DPROP_ALIAS_ABBREV_SINGULAR :
               DriverPrintResult("ALIAS ABBREV SINGULAR \n");
               break;
            case AR_DPROP_ALIAS_ABBREV_PLURAL :
               DriverPrintResult("ALIAS ABBREV PLURAL\n");
               break;
            case AR_DPROP_NAMED_SEARCHES  :
               DriverPrintResult("NAMED SEARCHES\n");
               break;
            case AR_DPROP_MENU_ACCESS     :
               DriverPrintResult("MENU ACCESS\n");
               break;
            case AR_DPROP_PANE_VISIBILITY_OPTION :
               DriverPrintResult("PANE VISIBILITY OPTION\n");
               break;
            case AR_DPROP_REQUEST_IDENTIFIER :
               DriverPrintResult("REQUEST IDENTIFIER\n");
               break;
            case AR_DPROP_QUERY_LIST_COLOR:
               DriverPrintResult("QUERY LIST COLOR\n");
               break;
            case AR_DPROP_COLUMN_WIDTH    :
               DriverPrintResult("COLUMN WIDTH\n");
               break;
            case AR_DPROP_COLUMN_ORDER    :
               DriverPrintResult("COLUMN ORDER\n");
               break;
            case AR_DPROP_SORT_SEQ        :
               DriverPrintResult("COLUMN SORT SEQUENCE\n");
               break;
            case AR_DPROP_SORT_DIR        :
               DriverPrintResult("COLUMN SORT DIRECTION\n");
               break;
            case AR_DPROP_DRILL_DOWN      :
               DriverPrintResult("DRILL DOWN\n");
               break;
            case AR_DPROP_REFRESH         :
               DriverPrintResult("REFRESH\n");
               break;
            case AR_DPROP_AUTO_REFRESH    :
               DriverPrintResult("AUTO REFRESH\n");
               break;
            case AR_DPROP_AUTOFIT_COLUMNS :
               DriverPrintResult("AUTOFIT COLUMNS\n");
               break;
            case AR_DPROP_APPLY_DIRTY     :
               DriverPrintResult("APPLY DIRTY\n");
               break;
            default                       :
               DriverPrintResult("   <undefined property = %u>\n",
                                 tempPtr->prop);
               break;
         }
         PrintARValueStruct(buff, &tempPtr->value);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARDisplayInstanceList                    */
/*                                                                           */
/*****************************************************************************/

void PrintARDisplayInstanceList(value)
ARDisplayInstanceList  *value;        /* IN; value to print */

{
   int                      i;        /* working index */
   ARDisplayInstanceStruct *tempPtr;  /* working pointer */

   DriverPrintResult("Display Instance List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      PrintARDisplayPropList("   ", "Common ", &value->commonProps);
      tempPtr = value->dInstanceList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         DriverPrintResult("   Display Instance : #%d for VUI %lu\n", i,
                           tempPtr->vui);
         PrintARDisplayPropList("   ", "Instance ", &tempPtr->props);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                      PrintARDisplayInstanceListList                       */
/*                                                                           */
/*****************************************************************************/

void PrintARDisplayInstanceListList(header, value)
char                      *header; /* IN; header string for the value */
ARDisplayInstanceListList *value;  /* IN; value to print */

{
   int      i;                /* working index */

   DriverPrintResult("%s : %u items\n", header, value->numItems);
   for (i = 0; i < (int) value->numItems; i++)
      PrintARDisplayInstanceList(&value->dInstanceList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARAssignFieldStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARAssignFieldStruct(header, value)
char                *header;  /* IN; header string for the value */
ARAssignFieldStruct *value;   /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%s", buff);
   PrintChar("Server: ", value->server);
   DriverPrintResult("%s", buff);
   PrintChar("Schema: ", value->schema);

   switch (value->tag)
   {
      case AR_FIELD        :
         DriverPrintResult("%sTag: FIELD\n", buff);
         DriverPrintResult("%s", buff);
         PrintARInternalId("Field Id:", value->u.fieldId);
         break;
      case AR_STAT_HISTORY :
         DriverPrintResult("%sTag: STAT_HISTORY\n", buff);
         DriverPrintResult("%s", buff);
         PrintLong("Enumerated value index:",
                   (int) value->u.statHistory.enumVal);
         switch (value->u.statHistory.userOrTime)
         {
            case AR_STAT_HISTORY_USER :
               DriverPrintResult("%sUser or Time: USER\n", buff);
               break;
            case AR_STAT_HISTORY_TIME :
               DriverPrintResult("%sUser or Time: TIME\n", buff);
               break;
            default                   :
               DriverPrintResult("%sUser or Time: %s%u>\n", buff,
                                 "<undefined value = ",
                                 value->u.statHistory.userOrTime);
               break;
         }
         break;
      default              :
         DriverPrintResult("%sTag: <undefined tag = %u>\n", buff, value->tag);
         break;
   }

   PrintARQualifierStruct(header, &value->qualifier);
   switch (value->noMatchOption)
   {
      case AR_NO_MATCH_ERROR    :
         DriverPrintResult("%sNo Match Option: ERROR\n", header);
         break;
      case AR_NO_MATCH_SET_NULL :
         DriverPrintResult("%sNo Match Option: NULL\n", header);
         break;
      case AR_NO_MATCH_NO_ACTION:
         DriverPrintResult("%sNo Match Option: NO ACTION\n", header);
         break;
      case AR_NO_MATCH_SUBMIT   :
         DriverPrintResult("%sNo Match Option: SUBMIT\n", header);
         break;
      default                   :
         DriverPrintResult("%sNo Match Option: %s%u>\n", header,
                           "<undefined value = ", value->noMatchOption);
         break;
   }
   switch (value->multiMatchOption)
   {
      case AR_MULTI_MATCH_ERROR    :
         DriverPrintResult("%sMulti Match Option: ERROR\n", header);
         break;
      case AR_MULTI_MATCH_SET_NULL :
         DriverPrintResult("%sMulti Match Option: NULL\n", header);
         break;
      case AR_MULTI_MATCH_USE_FIRST:
         DriverPrintResult("%sMulti Match Option: FIRST\n", header);
         break;
      case AR_MULTI_MATCH_PICKLIST :
         DriverPrintResult("%sMulti Match Option: PICKLIST\n", header);
         break;
      case AR_MULTI_MATCH_MODIFY_ALL :
         DriverPrintResult("%sMulti Match Option: MODIFY ALL\n", header);
         break;
      case AR_MULTI_MATCH_NO_ACTION :
         DriverPrintResult("%sMulti Match Option: NO ACTION\n", header);
         break;
      default                      :
         DriverPrintResult("%sMulti Match Option: %s%u>\n", header,
                           "<undefined value = ", value->multiMatchOption);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARAssignSQLStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARAssignSQLStruct(header, value)
char                *header;  /* IN; header string for the value */
ARAssignSQLStruct   *value;   /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%s", buff);
   PrintChar("Server: ", value->server);
   DriverPrintResult("%s", buff);
   if (value->sqlCommand == NULL)
      DriverPrintResult("SQL Command: NULL");
   else
      PrintChar("SQL Command: ", value->sqlCommand);

   DriverPrintResult("%s", buff);
   PrintInt("Value Index:", value->valueIndex);

   switch (value->noMatchOption)
   {
      case AR_NO_MATCH_ERROR    :
         DriverPrintResult("%sNo Match Option: ERROR\n", buff);
         break;
      case AR_NO_MATCH_SET_NULL :
         DriverPrintResult("%sNo Match Option: NULL\n", buff);
         break;
      default                   :
         DriverPrintResult("%sNo Match Option: %s%u>\n", buff,
                           "<undefined value = ", value->noMatchOption);
         break;
   }
   switch (value->multiMatchOption)
   {
      case AR_MULTI_MATCH_ERROR    :
         DriverPrintResult("%sMulti Match Option: ERROR\n", buff);
         break;
      case AR_MULTI_MATCH_SET_NULL :
         DriverPrintResult("%sMulti Match Option: NULL\n", buff);
         break;
      case AR_MULTI_MATCH_USE_FIRST:
         DriverPrintResult("%sMulti Match Option: FIRST\n", buff);
         break;
      case AR_MULTI_MATCH_PICKLIST :
         DriverPrintResult("%sMulti Match Option: PICKLIST\n", buff);
         break;
      default                      :
         DriverPrintResult("%sMulti Match Option: %s%u>\n", buff,
                           "<undefined value = ", value->multiMatchOption);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARAssignStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARAssignStruct(header, value)
char           *header;       /* IN; header string for the value */
ARAssignStruct *value;        /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%sAssign Value: ", buff);
   switch (value->assignType)
   {
      case AR_ASSIGN_TYPE_NONE    :
         DriverPrintResult("  NONE\n");
         break;
      case AR_ASSIGN_TYPE_VALUE   :
         DriverPrintResult("  VALUE\n");
         PrintARValueStruct(buff, &value->u.value);
         break;
      case AR_ASSIGN_TYPE_FIELD   :
         DriverPrintResult("  FIELD\n");
         PrintARAssignFieldStruct(buff, value->u.field);
         break;
      case AR_ASSIGN_TYPE_PROCESS :
         DriverPrintResult("  PROCESS\n");
         DriverPrintResult("%s", buff);
         PrintChar("Process: ", value->u.process);
         break;
      case AR_ASSIGN_TYPE_ARITH   :
         DriverPrintResult("  ARITH\n");
         PrintARArithOpAssignStruct(buff, value->u.arithOp);
         break;
      case AR_ASSIGN_TYPE_FUNCTION:
         DriverPrintResult("  FUNCTION\n");
         PrintARFunctionAssignStruct(buff, value->u.function);
         break;
      case AR_ASSIGN_TYPE_DDE     :
         DriverPrintResult("  DDE\n");
         PrintARDDEStruct(buff, value->u.dde);
         break;
      case AR_ASSIGN_TYPE_SQL     :
         DriverPrintResult("  SQL\n");
         PrintARAssignSQLStruct(buff, value->u.sql);
         break;
      default                   :
         DriverPrintResult("  <undefined assign type tag = %u>\n",
                           value->assignType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFieldAssignStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldAssignStruct(value)
ARFieldAssignStruct *value;   /* IN; value to print */

{
   DriverPrintResult("Field Assign Struct:\n");
   PrintARInternalId("   Field Id :", value->fieldId);
   PrintARAssignStruct("", &value->assignment);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFieldAssignList                           */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldAssignList(value)
ARFieldAssignList *value;     /* IN; value to print */

{
   int                  i;      /* working index */
   ARFieldAssignStruct *tempPtr;/* working pointer */

   DriverPrintResult("Field Assign List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->fieldAssignList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFieldAssignStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARArithOpAssignStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARArithOpAssignStruct(header, value)
char                  *header; /* IN; header string for the value */
ARArithOpAssignStruct *value;  /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%sArithmetic Operation:\n", header);
   switch (value->operation)
   {
      case AR_ARITH_OP_ADD      :
         DriverPrintResult("%s   Operation: ADD\n", header);
         break;
      case AR_ARITH_OP_SUBTRACT :
         DriverPrintResult("%s   Operation: SUBTRACT\n", header);
         break;
      case AR_ARITH_OP_MULTIPLY :
         DriverPrintResult("%s   Operation: MULTIPLY\n", header);
         break;
      case AR_ARITH_OP_DIVIDE   :
         DriverPrintResult("%s   Operation: DIVIDE\n", header);
         break;
      case AR_ARITH_OP_MODULO   :
         DriverPrintResult("%s   Operation: MODULO\n", header);
         break;
      case AR_ARITH_OP_NEGATE   :
         DriverPrintResult("%s   Operation: NEGATE (unary minus)\n", header);
         break;
      default                   :
         DriverPrintResult("%s   Operation: <undefined operation = %u>\n",
                           header, value->operation);
         break;
   }
   if (value->operation != AR_ARITH_OP_NEGATE)
      PrintARAssignStruct(buff, &value->operandLeft);
   PrintARAssignStruct(buff, &value->operandRight);
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARFunctionAssignStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARFunctionAssignStruct(header, value)
char                   *header; /* IN; header string for the value */
ARFunctionAssignStruct *value;  /* IN; value to print */

{
   char   buff[45];           /* buffer to build nested header in */
   int    i;                  /* working index */

   if (strlen(header) >= (size_t) 40)
      buff[0] = '\0';
   else
      (void) sprintf(buff, "%s   ", header);

   DriverPrintResult("%sFunction:\n", header);
   switch (value->functionCode)
   {
      case AR_FUNCTION_DATE     :
         DriverPrintResult("%s   Function code: DATE\n", header);
         break;
      case AR_FUNCTION_TIME     :
         DriverPrintResult("%s   Function code: TIME\n", header);
         break;
      case AR_FUNCTION_MONTH    :
         DriverPrintResult("%s   Function code: MONTH\n", header);
         break;
      case AR_FUNCTION_DAY      :
         DriverPrintResult("%s   Function code: DAY\n", header);
         break;
      case AR_FUNCTION_YEAR     :
         DriverPrintResult("%s   Function code: YEAR\n", header);
         break;
      case AR_FUNCTION_WEEKDAY  :
         DriverPrintResult("%s   Function code: WEEKDAY\n", header);
         break;
      case AR_FUNCTION_HOUR     :
         DriverPrintResult("%s   Function code: HOUR\n", header);
         break;
      case AR_FUNCTION_MINUTE   :
         DriverPrintResult("%s   Function code: MINUTE\n", header);
         break;
      case AR_FUNCTION_SECOND   :
         DriverPrintResult("%s   Function code: SECOND\n", header);
         break;
      case AR_FUNCTION_TRUNC    :
         DriverPrintResult("%s   Function code: TRUNC\n", header);
         break;
      case AR_FUNCTION_ROUND    :
         DriverPrintResult("%s   Function code: ROUND\n", header);
         break;
      case AR_FUNCTION_CONVERT  :
         DriverPrintResult("%s   Function code: CONVERT\n", header);
         break;
      case AR_FUNCTION_LENGTH   :
         DriverPrintResult("%s   Function code: LENGTH\n", header);
         break;
      case AR_FUNCTION_UPPER    :
         DriverPrintResult("%s   Function code: UPPER\n", header);
         break;
      case AR_FUNCTION_LOWER    :
         DriverPrintResult("%s   Function code: LOWER\n", header);
         break;
      case AR_FUNCTION_SUBSTR   :
         DriverPrintResult("%s   Function code: SUBSTR\n", header);
         break;
      case AR_FUNCTION_LEFT     :
         DriverPrintResult("%s   Function code: LEFT\n", header);
         break;
      case AR_FUNCTION_RIGHT    :
         DriverPrintResult("%s   Function code: RIGHT\n", header);
         break;
      case AR_FUNCTION_LTRIM    :
         DriverPrintResult("%s   Function code: LTRIM\n", header);
         break;
      case AR_FUNCTION_RTRIM    :
         DriverPrintResult("%s   Function code: RTRIM\n", header);
         break;
      case AR_FUNCTION_LPAD     :
         DriverPrintResult("%s   Function code: LPAD\n", header);
         break;
      case AR_FUNCTION_RPAD     :
         DriverPrintResult("%s   Function code: RPAD\n", header);
         break;
      case AR_FUNCTION_REPLACE  :
         DriverPrintResult("%s   Function code: REPLACE\n", header);
         break;
      case AR_FUNCTION_STRSTR   :
         DriverPrintResult("%s   Function code: STRSTR\n", header);
         break;
      case AR_FUNCTION_MIN      :
         DriverPrintResult("%s   Function code: MIN\n", header);
         break;
      case AR_FUNCTION_MAX      :
         DriverPrintResult("%s   Function code: MAX\n", header);
         break;
      default                   :
         DriverPrintResult("%s   Function code: <undefined function = %u>\n",
                           header, value->functionCode);
         break;
   }
   DriverPrintResult("%s   Number parameters: %u\n", header, value->numItems);

   if ((value->numItems != 0) && (value->parameterList != NULL))
      for (i = 0; i < (int) value->numItems; i++)
         PrintARAssignStruct(buff, &value->parameterList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFilterActionNotify                        */
/*                                                                           */
/*****************************************************************************/

void PrintARFilterActionNotify(value)
ARFilterActionNotify *value;  /* IN; value to print */

{
   DriverPrintResult("NOTIFY\n");
   PrintChar("   User:", value->user);
   PrintInt("   Priority:", value->notifyPriority);
   PrintChar("   Text:", value->notifyText);
   PrintChar("   Subject:", value->subjectText);
   switch (value->notifyMechanism)
   {
      case AR_NOTIFY_VIA_NOTIFIER :
         DriverPrintResult("   Mechanism: NOTIFIER\n");
         break;
      case AR_NOTIFY_VIA_EMAIL    :
         DriverPrintResult("   Mechanism: E-MAIL\n");
         break;
      case AR_NOTIFY_VIA_DEFAULT  :
         DriverPrintResult("   Mechanism: DEFAULT\n");
         break;
      case AR_NOTIFY_VIA_XREF     :
         DriverPrintResult("   Mechanism: CROSS-REFERENCE\n");
         DriverPrintResult("   Cross reference field id: %ld",
                           value->notifyMechanismXRef);
         break;
      default                     :
         DriverPrintResult("   Mechanism: %u\n", value->notifyMechanism);
         break;
   }
   switch (value->fieldIdListType)
   {
      case AR_FILTER_FIELD_IDS_NONE :
         DriverPrintResult("   Field Id List: NONE\n");
         break;
      case AR_FILTER_FIELD_IDS_ALL  :
         DriverPrintResult("   Field Id List: ALL\n");
         break;
      case AR_FILTER_FIELD_IDS_LIST :
         DriverPrintResult("   Field Id List: LIST\n");
         PrintARInternalIdList("      Field List:", "         ",
                               &value->fieldIdList);
         break;
      case AR_FILTER_FIELD_IDS_CHANGED :
         DriverPrintResult("   Field Id List: CHANGED\n");
         break;
      default                     :
         DriverPrintResult("   Field Id List: <undefined type = %u>\n",
                           value->fieldIdListType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFilterStatusStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARFilterStatusStruct(value)
ARFilterStatusStruct *value;        /* IN; value to print */

{
   DriverPrintResult("Filter Status Struct :\n");
   DriverPrintResult("   Message type : ");
   switch (value->messageType)
   {
      case AR_RETURN_OK         :
         DriverPrintResult("NOTE\n");
         break;
      case AR_RETURN_WARNING    :
         DriverPrintResult("WARNING\n");
         break;
      case AR_RETURN_ERROR      :
         DriverPrintResult("ERROR\n");
         break;
      default                   :
         DriverPrintResult("<unknown type>\n");
         break;
   }
   PrintLong("   Message number : ", value->messageNum);
   PrintChar("   Message: ", value->messageText);
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintARPushFieldsStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARPushFieldsStruct(value)
ARPushFieldsStruct *value;  /* IN; value to print */

{
   DriverPrintResult("Push Fields Struct:\n");
   PrintARAssignFieldStruct("", &value->field);
   PrintARAssignStruct("", &value->assign);
}

/*****************************************************************************/
/*                                                                           */
/*                            PrintARPushFieldsList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARPushFieldsList(value)
ARPushFieldsList *value;  /* IN; value to print */

{
   unsigned int    i;    /* working index */

   if ((value->numItems != 0) && (value->pushFieldsList != NULL))
   {
      DriverPrintResult("Push Fields List : %u items\n", value->numItems);
      for (i = 0; i < value->numItems; i++)
         PrintARPushFieldsStruct(&value->pushFieldsList[i]);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                               PrintARSQLStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARSQLStruct(value)
ARSQLStruct *value;  /* IN; value to print */

{
   DriverPrintResult("SQL Struct:\n");
   PrintChar("Server: ", value->server);
   if (value->command == NULL)
      DriverPrintResult("SQL Command: NULL");
   else
      PrintChar("SQL Command: ", value->command);
}

/*****************************************************************************/
/*                                                                           */
/*                       PrintARGotoActionStruct                             */
/*                                                                           */
/*****************************************************************************/
  
void PrintARGotoActionStruct(value)
ARGotoActionStruct *value;
{
   DriverPrintResult("   Goto Action Struct:\n");
   PrintInt("      Tag (Field Or Value): ", value->tag);
   if (value->tag == AR_GOTO_FIELD_XREF)
      PrintInt("      FieldId: ", value->fieldIdOrValue);
   else if (value->tag == AR_GOTO_ABSOLUTE_ORDER)
      PrintInt("      Execution Order: ", value->fieldIdOrValue);
   else if (value->tag == AR_GOTO_OFFSET_FORWARD)
      PrintInt("      Execution Order Advance By: ", value->fieldIdOrValue);
   else if (value->tag == AR_GOTO_OFFSET_BACKWARD)
      PrintInt("      Execution Order Backup By: ", value->fieldIdOrValue);
   else
      DriverPrintResult("<unknown type>\n");
}

/*****************************************************************************/
/*                                                                           */
/*                         PrintARFilterActionStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARFilterActionStruct(value)
ARFilterActionStruct *value;  /* IN; value to print */

{
   DriverPrintResult("Filter Action Struct:  ");
   switch (value->action)
   {
      case AR_FILTER_ACTION_NOTIFY  :
         PrintARFilterActionNotify(&value->u.notify);
         break;
      case AR_FILTER_ACTION_MESSAGE :
         DriverPrintResult("MESSAGE\n");
         PrintARFilterStatusStruct(&value->u.message);
         break;
      case AR_FILTER_ACTION_LOG     :
         DriverPrintResult("LOG FILE\n");
         PrintChar("   Filename:", value->u.logFile);
         break;
      case AR_FILTER_ACTION_FIELDS  :
         DriverPrintResult("SET FIELDS\n");
         PrintARFieldAssignList(&(value->u.fieldList));
         break;
      case AR_FILTER_ACTION_PROCESS :
         DriverPrintResult("PROCESS\n");
         PrintChar("   Filename:", value->u.process);
         break;
      case AR_FILTER_ACTION_FIELDP  :
         DriverPrintResult("PUSH FIELDS\n");
         PrintARPushFieldsList(&(value->u.pushFieldsList));
         break;
      case AR_FILTER_ACTION_SQL     :
         DriverPrintResult("DIRECT SQL\n");
         PrintARSQLStruct(&(value->u.sqlCommand));
         break;
      case AR_FILTER_ACTION_GOTOACTION     :
         DriverPrintResult("GOTO ACTION\n");
         PrintARGotoActionStruct(&(value->u.gotoAction));
         break;
      default                       :
         DriverPrintResult("   <undefined action = %u>\n", value->action);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARFilterActionList                           */
/*                                                                           */
/*****************************************************************************/

void PrintARFilterActionList(value, actionFlag)
ARFilterActionList *value;     /* IN; value to print */
ARBoolean           actionFlag;/* IN; TRUE - action; FALSE - else */

{
   int                   i;       /* working index */
   ARFilterActionStruct *tempPtr;/* working pointer */

   if (actionFlag)
      DriverPrintResult("Filter Action List : %u items\n", value->numItems);
   else
      DriverPrintResult("Filter Else List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->actionList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFilterActionStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARActiveLinkMacroStruct                      */
/*                                                                           */
/*****************************************************************************/

void PrintARActiveLinkMacroStruct(value)
ARActiveLinkMacroStruct *value;  /* IN; value to print */

{
   int   i;                   /* working index */

   DriverPrintResult("   Active Link Macro Struct:\n");
   PrintChar("      Macro name:", value->macroName);
   PrintChar("      Macro text:", value->macroText);
   DriverPrintResult("      Macro Parms:  %u\n", value->macroParms.numItems);
   for (i = 0; i < (int) value->macroParms.numItems; i++)
   {
      DriverPrintResult("      Macro parm (%d): %s", (i + 1),
                  value->macroParms.parms[i].name);
      if (value->macroParms.parms[i].value == NULL)
         DriverPrintResult("\n");
      else
         DriverPrintResult("  =  %s\n", value->macroParms.parms[i].value);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                            PrintARMessageStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARMessageStruct(value)
ARMessageStruct *value;        /* IN; value to print */

{
   DriverPrintResult("Message Struct :\n");
   DriverPrintResult("   Message type : ");
   switch (value->messageType)
   {
      case AR_RETURN_OK         :
         DriverPrintResult("NOTE\n");
         break;
      case AR_RETURN_WARNING    :
         DriverPrintResult("WARNING\n");
         break;
      case AR_RETURN_ERROR      :
         DriverPrintResult("ERROR\n");
         break;
      case AR_RETURN_PROMPT     :
         DriverPrintResult("PROMPT\n");
         break;
      default                   :
         DriverPrintResult("<unknown type>\n");
         break;
   }
   PrintLong("   Message number : ", value->messageNum);
   PrintChar("   Message: ", value->messageText);
   PrintARBoolean("   Use Prompting Pane : ", value->usePromptingPane);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARFieldCharacteristics                       */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldCharacteristics(value)
ARFieldCharacteristics *value;  /* IN; value to print */

{
   DriverPrintResult("   Active Link Field Characteristics Struct:\n");
   PrintLong("      Field Id:", value->fieldId);
   if (value->charMenu == NULL)
      DriverPrintResult("      Char menu: NULL\n");
   else
      PrintChar("      Char menu:", value->charMenu);
   PrintARDisplayPropList("      ", "", &value->props);
   PrintInt("      Reset Focus:", value->focus);
   PrintInt("      Access option:", value->accessOption);
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARDDEStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARDDEStruct(header, value)
char          *header;        /* IN; header string for the value */
ARDDEStruct   *value;         /* IN; value to print */

{
   DriverPrintResult("   Active Link DDE Struct:\n");
   PrintChar("      Service name:", value->serviceName);
   PrintChar("      Topic:", value->topic);
   PrintInt("      Action:", value->action);
   PrintChar("      Path to Program:", value->pathToProgram);
   PrintChar("      Command:", value->command);
   if (value->item != NULL)
      PrintChar("      Item:", value->item);
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARAutomationStruct                     */
/*                                                                           */
/*****************************************************************************/
 
void PrintARAutomationStruct(header, value)
char                 *header;        /* IN; header string for the value */
ARAutomationStruct   *value;         /* IN; value to print */
 
{
   DriverPrintResult("   Active Link Outbound Automation Struct:\n");
   PrintChar("      Auto Server name:", value->autoServerName);
   PrintARBoolean("      Automation Server Showing flag :", value->isVisible);
   PrintChar("      Class Id:", value->clsId);
   PrintChar("      Automation Action:", value->action);
   PrintARCOMMethodList(&value->methodList);
}
 
/*****************************************************************************/
/*                                                                           */
/*                               PrintARCOMValueStruct                       */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMValueStruct(header, value, indent)
char             *header;       /* IN; header string for the value */
ARCOMValueStruct *value;        /* IN; value to print */
char             *indent;       /* IN; for indentation */
 
{
   char buffer[512];

   DriverPrintResult("      %sCOM %s Value Struct:\n", indent, header);
   sprintf(buffer, "         %sValue IId:", indent);
   PrintChar(buffer, ((value->valueIId)? value->valueIId : "(NULL)"));
   sprintf(buffer, "         %sTransient Id:", indent);
   PrintLong(buffer, value->transId);

   switch (value->valueType)
   {
      case AR_COM_PARM_NULL:
         sprintf(buffer, "         %sValue: <NULL>", indent);
         PrintChar(buffer, "");
         break;
      case AR_COM_PARM_FIELDID: /* AR_COM_METHOD_FIELDID has same type */
         sprintf(buffer, "         %sFieldId:", indent);
         PrintLong(buffer, value->u.fieldId);
         break;
      case AR_COM_PARM_VALUE:
         sprintf(buffer, "         %s", indent);
         PrintARValueStruct(buffer, &value->u.value);
         break;
      default:
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARCOMMethodList                        */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMMethodList(value)
ARCOMMethodList     *value;         /* IN; value to print */
 
{
   int                   i;
   ARCOMMethodStruct    *tempPtr;
 
   DriverPrintResult("    COM Automation Method List:\n");
   if (value->numItems > 0)
   {
      tempPtr = value->methodList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARCOMMethodStruct(tempPtr);
         tempPtr++;
      }
   }
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                               PrintARCOMMethodStruct                      */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMMethodStruct(value)
ARCOMMethodStruct       *value;         /* IN; value to print */
 
{
   DriverPrintResult("      COM Method Struct:\n");
   PrintChar("      Method Name:", value->methodName);
   PrintChar("      Method Interface Id:", value->methodIId);
   PrintInt("      Method Type:", value->methodType);
   PrintARCOMValueStruct("Method", &value->methodValue, "");
   PrintARCOMMethodParmList(&(value->parameterList));
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                        PrintARCOMMethodParmList                           */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMMethodParmList(value)
ARCOMMethodParmList     *value;         /* IN; value to print */
 
{
   int                           i;
   ARCOMMethodParmStruct        *tempPtr;
 
   DriverPrintResult("      COM Parameter List:\n");
   if (value->numItems > 0)
   {
      tempPtr = value->parameterList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARCOMMethodParmStruct(tempPtr);
         tempPtr++;
      }
   }
}
 
 
/*****************************************************************************/
/*                                                                           */
/*                               PrintARCOMMethodParmStruct                  */
/*                                                                           */
/*****************************************************************************/
 
void PrintARCOMMethodParmStruct(value)
ARCOMMethodParmStruct   *value;         /* IN; value to print */
 
{
   DriverPrintResult("         COM Parameter Struct:\n");
   PrintChar("         Parameter Name:", value->parmName);
   PrintInt("         Parameter Type:", value->parmType);
   PrintARCOMValueStruct("Parameter", &value->parmValue, "   ");
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintAROpenDlgStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintAROpenDlgStruct(value)
AROpenDlgStruct   *value;      /* OUT; loaded with value for buffer */
 
{
   DriverPrintResult("   Active Link Open Dialog Struct:\n");
   PrintChar("      Server Name: ", value->serverName);
   PrintChar("      Schema Name: ", value->schemaName);
   PrintChar("      Vui Label: ", value->vuiLabel);
   PrintARBoolean("      Close Box Showing flag : ", value->closeBox);
   PrintARFieldAssignList(&value->inputValueFieldPairs);
   PrintARFieldAssignList(&value->outputValueFieldPairs);
}

  
/*****************************************************************************/
/*                                                                           */
/*                             PrintARCallGuideStruct                        */
/*                                                                           */
/*****************************************************************************/
  
void PrintARCallGuideStruct(value)
ARCallGuideStruct *value;

{
   DriverPrintResult("   Active Link Call Guide Struct:\n");
   PrintChar("      Server Name: ", value->serverName);
   PrintChar("      Guide Name: ", value->guideName);
   PrintInt("      Guide Mode: ", value->guideMode);
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARExitGuideStruct                        */
/*                                                                           */
/*****************************************************************************/
  
void PrintARExitGuideStruct(value)
ARExitGuideStruct *value;

{
   DriverPrintResult("   Active Link Exit Guide Struct:\n");
   PrintARBoolean("      Close All Windows : ", value->closeAll);
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARGotoGuideLabel                         */
/*                                                                           */
/*****************************************************************************/
  
void PrintARGotoGuideLabel(value)
ARGotoGuideLabelStruct *value;

{
   DriverPrintResult("   Active Link Goto Guide Label Struct:\n");
   PrintChar("      Label: ", value->label);
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARWaitStruct                             */
/*                                                                           */
/*****************************************************************************/
  
void PrintARWaitStruct(value)
ARWaitStruct *value;

{
   DriverPrintResult("   Active Link Wait Struct:\n");
   PrintChar("      Continue Button Title: ", value->continueButtonTitle);
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARActiveLinkActionStruct                      */
/*                                                                           */
/*****************************************************************************/

void PrintARActiveLinkActionStruct(value)
ARActiveLinkActionStruct *value;  /* IN; value to print */

{
   DriverPrintResult("Active Link Action Struct:  ");
   switch (value->action)
   {
      case AR_ACTIVE_LINK_ACTION_MACRO   :
         DriverPrintResult("MACRO\n");
         PrintARActiveLinkMacroStruct(&value->u.macro);
         break;
      case AR_ACTIVE_LINK_ACTION_FIELDS  :
         DriverPrintResult("FIELDS\n");
         PrintARFieldAssignList(&value->u.fieldList);
         break;
      case AR_ACTIVE_LINK_ACTION_PROCESS :
         DriverPrintResult("PROCESS\n");
         PrintChar("   Process:", value->u.process);
         break;
      case AR_ACTIVE_LINK_ACTION_MESSAGE :
         DriverPrintResult("MESSAGE\n");
         PrintARMessageStruct(&value->u.message);
         break;
      case AR_ACTIVE_LINK_ACTION_SET_CHAR:
         DriverPrintResult("SET FIELD CHARACTERISTICS\n");
         PrintARFieldCharacteristics(&value->u.characteristics);
         break;
      case AR_ACTIVE_LINK_ACTION_DDE     :
         DriverPrintResult("DDE\n");
         PrintARDDEStruct("      ", &value->u.dde);
         break;
      case AR_ACTIVE_LINK_ACTION_FIELDP  :
         DriverPrintResult("PUSH FIELDS\n");
         PrintARPushFieldsList(&(value->u.pushFieldsList));
         break;
      case AR_ACTIVE_LINK_ACTION_SQL     :
         DriverPrintResult("DIRECT SQL\n");
         PrintARSQLStruct(&(value->u.sqlCommand));
         break;
      case AR_ACTIVE_LINK_ACTION_AUTO    :
         DriverPrintResult("Automation\n");
         PrintARAutomationStruct("      ", &value->u.automation);
         break;
      case AR_ACTIVE_LINK_ACTION_OPENDLG   :
         DriverPrintResult("Open Dialog\n");
         PrintAROpenDlgStruct(&value->u.openDlg);
         break;
      case AR_ACTIVE_LINK_ACTION_COMMITC   :
         DriverPrintResult("Commit Changes\n");
         break;
      case AR_ACTIVE_LINK_ACTION_CLOSEWND :
         DriverPrintResult("Close Window\n");
         break;
      case AR_ACTIVE_LINK_ACTION_CALLGUIDE  :
         DriverPrintResult("Call Guide\n");
         PrintARCallGuideStruct(&value->u.callGuide);
         break;
      case AR_ACTIVE_LINK_ACTION_EXITGUIDE  :
         DriverPrintResult("Exit Guide\n");
         PrintARExitGuideStruct(&value->u.exitGuide);
         break;
      case AR_ACTIVE_LINK_ACTION_GOTOGUIDELABEL :
         DriverPrintResult("Goto Guide\n");
         PrintARGotoGuideLabel(&value->u.gotoGuide);
         break;
      case AR_ACTIVE_LINK_ACTION_WAIT:
         DriverPrintResult("Wait Guide\n");
         PrintARWaitStruct(&value->u.waitAction);
         break;
      case AR_ACTIVE_LINK_ACTION_GOTOACTION:
         DriverPrintResult("Goto Action\n");
         PrintARGotoActionStruct(&value->u.gotoAction);
         break;
      default          :
         DriverPrintResult("   <undefined action = %u>\n", value->action);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARActiveLinkActionList                       */
/*                                                                           */
/*****************************************************************************/

void PrintARActiveLinkActionList(value, actionFlag)
ARActiveLinkActionList *value;     /* IN; value to print */
ARBoolean               actionFlag;/* IN; TRUE - action; FALSE - else */

{
   int                       i;      /* working index */
   ARActiveLinkActionStruct *tempPtr;/* working pointer */

   if (actionFlag)
      DriverPrintResult("Active Link Action List : %u items\n",
                        value->numItems);
   else
      DriverPrintResult("Active Link Else List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->actionList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARActiveLinkActionStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARPermissionStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARPermissionStruct(value, fieldFlag)
ARPermissionStruct *value;    /* IN; value to print */
ARBoolean           fieldFlag;/* IN; TRUE - field; FALSE - schema */

{
   DriverPrintResult("Permission Struct:\n");
   PrintARInternalId("   Group id:", value->groupId);
   switch (value->permissions)
   {
      case AR_PERMISSIONS_NONE   :
         DriverPrintResult("   Permissions:  NONE\n");
         break;
      case AR_PERMISSIONS_VIEW   :
         if (fieldFlag)
            DriverPrintResult("   Permissions:  VIEW\n");
         else
            DriverPrintResult("   Permissions:  VISIBLE\n");
         break;
      case AR_PERMISSIONS_CHANGE :
         if (fieldFlag)
            DriverPrintResult("   Permissions:  CHANGE\n");
         else
            DriverPrintResult("   Permissions:  HIDDEN\n");
         break;
      default                    :
         DriverPrintResult("   Permissions: <undefined permission = %u>\n",
                           value->permissions);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARPermissionList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARPermissionList(value, fieldFlag)
ARPermissionList *value;      /* IN; value to print */
ARBoolean         fieldFlag;  /* IN; TRUE - field; FALSE - schema */

{
   int                 i;      /* working index */
   ARPermissionStruct *tempPtr;/* working pointer */

   DriverPrintResult("Permission List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->permissionList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARPermissionStruct(tempPtr, fieldFlag);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARPermissionListList                         */
/*                                                                           */
/*****************************************************************************/

void PrintARPermissionListList(header, value)
char                 *header; /* IN; header string for the value */
ARPermissionListList *value;  /* IN; value to print */

{
   int      i;                /* working index */

   DriverPrintResult("%s : %u items\n", header, value->numItems);
   for (i = 0; i < (int) value->numItems; i++)
      PrintARPermissionList(&value->permissionList[i], TRUE);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARGroupInfoStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARGroupInfoStruct(value)
ARGroupInfoStruct  *value;    /* IN; value to print */

{
   DriverPrintResult("   Group Info Struct:\n");
   PrintARInternalId("      Group id:", value->groupId);
   switch (value->groupType)
   {
      case 0                    :
         DriverPrintResult("      Group type:  NONE\n");
         break;
      case AR_GROUP_TYPE_VIEW   :
         DriverPrintResult("      Group type:  VIEW\n");
         break;
      case AR_GROUP_TYPE_CHANGE :
         DriverPrintResult("      Group type:  CHANGE\n");
         break;
      default                    :
         DriverPrintResult("      Group type:  <undefined group type = %u>\n",
                           value->groupType);
         break;
   }
   PrintARNameList("      Group name(s): ", "      ", &value->groupName);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARGroupInfoList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARGroupInfoList(value)
ARGroupInfoList *value;       /* IN; value to print */

{
   int                i;       /* working index */
   ARGroupInfoStruct *tempPtr; /* working pointer */

   DriverPrintResult("Group Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->groupList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARGroupInfoStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARUserLicenseStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARUserLicenseStruct(value)
ARUserLicenseStruct *value;   /* IN; value to print */

{
   char *tempPtr;             /* working pointer */
   char *tempPtr2;            /* working pointer */

   DriverPrintResult("         User License Struct:  ");
   switch (value->licenseTag)
   {
      case AR_LICENSE_TAG_WRITE          :
         DriverPrintResult("WRITE\n");
         switch (value->licenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr = "READ";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr = "FLOATING";
               break;
            default                    :
               tempPtr = "<undefined>";
               break;
         }
         switch (value->currentLicenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr2 = "READ";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr2 = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr2 = "FLOATING";
               break;
            default                    :
               tempPtr2 = "<undefined>";
               break;
         }
         break;
      case AR_LICENSE_TAG_FULL_TEXT      :
         DriverPrintResult("FULL TEXT\n");
         switch (value->licenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr = "NONE";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr = "FLOATING";
               break;
            default                    :
               tempPtr = "<undefined>";
               break;
         }
         switch (value->currentLicenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr2 = "NONE";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr2 = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr2 = "FLOATING";
               break;
            default                    :
               tempPtr2 = "<undefined>";
               break;
         }
         break;
      case AR_LICENSE_TAG_RESERVED1     :
         DriverPrintResult("RESERVED\n");
         switch (value->licenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr = "NONE";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr = "FLOATING";
               break;
            default                    :
               tempPtr = "<undefined>";
               break;
         }
         switch (value->currentLicenseType)
         {
            case AR_LICENSE_TYPE_NONE :
               tempPtr2 = "NONE";
               break;
            case AR_LICENSE_TYPE_FIXED :
               tempPtr2 = "FIXED";
               break;
            case AR_LICENSE_TYPE_FLOATING :
               tempPtr2 = "FLOATING";
               break;
            default                    :
               tempPtr2 = "<undefined>";
               break;
         }
         break;
      default                       :
         DriverPrintResult("   <undefined operation = %u>\n", value->licenseTag);
         tempPtr = "";
         tempPtr2 = "";
         break;
   }
   PrintChar("            Assigned License Type:", tempPtr);
   PrintChar("            Current License Type:", tempPtr2);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARUserLicenseList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARUserLicenseList(value)
ARUserLicenseList *value;       /* IN; value to print */

{
   int                  i;       /* working index */
   ARUserLicenseStruct *tempPtr; /* working pointer */

   DriverPrintResult("      User License List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->licenseList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARUserLicenseStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARUserInfoStruct                            */
/*                                                                           */
/*****************************************************************************/

void PrintARUserInfoStruct(value)
ARUserInfoStruct  *value;    /* IN; value to print */

{
   DriverPrintResult("   User Info Struct:\n");
   PrintChar("      User name:", value->userName);
   PrintARUserLicenseList(&value->licenseInfo);
   PrintARTimestamp("      Connect time:", value->connectTime);
   PrintARTimestamp("      Last access:", value->lastAccess);
   PrintChar("      Email address:", value->emailAddr);
   PrintInt("      Default notify mechanism:", value->defaultNotifyMech);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARUserInfoList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARUserInfoList(value)
ARUserInfoList *value;       /* IN; value to print */

{
   int               i;       /* working index */
   ARUserInfoStruct *tempPtr; /* working pointer */

   DriverPrintResult("User Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->userList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARUserInfoStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARIntegerLimitsStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARIntegerLimitsStruct(value)
ARIntegerLimitsStruct *value; /* IN; value to print */

{
   DriverPrintResult("INTEGER\n");
   PrintLong("   Low range :", (int)value->rangeLow);
   PrintLong("   High range:", (int)value->rangeHigh);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARRealLimitsStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARRealLimitsStruct(value)
ARRealLimitsStruct *value;    /* IN; value to print */

{
   DriverPrintResult("REAL\n");
   PrintReal("   Low range :", value->rangeLow);
   PrintReal("   High range:", value->rangeHigh);
   PrintInt("   Precision:", value->precision);
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARCharLimitsStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARCharLimitsStruct(value)
ARCharLimitsStruct *value;    /* IN; value to print */

{
   DriverPrintResult("CHARACTER\n");
   PrintInt("   Maximum length:", value->maxLength);
   PrintInt("   Menu style:", value->menuStyle);
   PrintInt("   QBE match operation:", value->qbeMatchOperation);
   PrintChar("   Character menu:", value->charMenu);
   PrintChar("   Pattern:", value->pattern);
   PrintInt("   FullText Options:", value->fullTextOptions);
}

/*****************************************************************************/
/*                                                                           */
/*                          PrintARDiaryLimitsStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARDiaryLimitsStruct(value)
ARDiaryLimitsStruct *value;    /* IN; value to print */

{
   DriverPrintResult("DIARY\n");
   PrintInt("   FullText Options:", value->fullTextOptions);
}

/*****************************************************************************/
/*                                                                           */
/*                          PrintARAttachLimitsStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARAttachLimitsStruct(value)
ARAttachLimitsStruct *value;    /* IN; value to print */

{
   DriverPrintResult("ATTACHMENT\n");
   PrintInt("   Maximum size:", value->maxSize);
   PrintInt("   Attachment Type:", value->attachType);
}

/*****************************************************************************/
/*                                                                           */
/*                          PrintARTableLimitsStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARTableLimitsStruct(header, value)
char                *header;   /* IN; header of the value */
ARTableLimitsStruct *value;    /* IN; value to print */
{
   DriverPrintResult("%s\n", header);
   PrintChar("   Schema:", value->schema);
   PrintChar("   Server:", value->server);
   PrintARQualifierStruct("   ", &value->qualifier);
   PrintInt("   numColumns:", value->numColumns);
   PrintInt("   maxRetrieve:", value->maxRetrieve);
}

/*****************************************************************************/
/*                                                                           */
/*                          PrintARColumnLimitsStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARColumnLimitsStruct(header, value)
char                *header;   /* IN; header of the value */
ARColumnLimitsStruct *value;    /* IN; value to print */

{
   DriverPrintResult("%s\n", header);
   PrintInt("   parent:", value->parent);
   PrintInt("   dataField:", value->dataField);
   PrintInt("   colLength:", value->colLength);
}

/*****************************************************************************/
/*                                                                           */
/*                         PrintARDecimalLimitsStruct                        */
/*                                                                           */
/*****************************************************************************/

void PrintARDecimalLimitsStruct(value)
ARDecimalLimitsStruct *value;    /* IN; value to print */

{
   DriverPrintResult("DECIMAL\n");
   PrintChar("   Low range :", value->rangeLow);
   PrintChar("   High range:", value->rangeHigh);
   PrintInt("   Precision:", value->precision);
}

/*****************************************************************************/
/*                                                                           */
/*                         PrintARFieldLimitStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldLimitStruct(value)
ARFieldLimitStruct *value;    /* IN; value to print */

{
   DriverPrintResult("Field Limit Struct:  ");
   switch (value->dataType)
   {
      case AR_FIELD_LIMIT_NONE  :
         DriverPrintResult("   NO LIMITS\n");
         break;
      case AR_DATA_TYPE_INTEGER :
         PrintARIntegerLimitsStruct(&value->u.intLimits);
         break;
      case AR_DATA_TYPE_REAL    :
         PrintARRealLimitsStruct(&value->u.realLimits);
         break;
      case AR_DATA_TYPE_CHAR    :
         PrintARCharLimitsStruct(&value->u.charLimits);
         break;
      case AR_DATA_TYPE_DIARY   :
         PrintARDiaryLimitsStruct(&value->u.diaryLimits);
         break;
      case AR_DATA_TYPE_ENUM    :
         DriverPrintResult("ENUMERATED\n");
         PrintARNameList("   Enum values:", "      ", &value->u.enumLimits);
         break;
      case AR_DATA_TYPE_TIME    :
         DriverPrintResult("TIMESTAMP\n");
         break;
      case AR_DATA_TYPE_BITMASK :
         DriverPrintResult("BITMASK\n");
         PrintARNameList("   Bitmask values:", "      ", &value->u.maskLimits);
         break;
      case AR_DATA_TYPE_ATTACH :
         PrintARAttachLimitsStruct(&value->u.attachLimits);
         break;
      case AR_DATA_TYPE_DECIMAL:
         PrintARDecimalLimitsStruct(&value->u.decimalLimits);
         break;
      case AR_DATA_TYPE_TABLE   :
         PrintARTableLimitsStruct("   Table values:", &value->u.tableLimits);
         break;
      case AR_DATA_TYPE_COLUMN  :
         PrintARColumnLimitsStruct("   Column values:", &value->u.columnLimits);
         break;
      default                   :
         DriverPrintResult("   <undefined datatype = %u>\n", value->dataType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                          PrintARFieldLimitList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldLimitList(header, value)
char              *header;    /* IN; header string for the value */
ARFieldLimitList  *value;     /* IN; value to print */

{
   int      i;                /* working index */

   DriverPrintResult("%s : %u items\n", header, value->numItems);
   for (i = 0; i < (int) value->numItems; i++)
      PrintARFieldLimitStruct(&value->fieldLimitList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARCharMenuItemStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuItemStruct(indent, value)
char                 *indent;     /* IN; string of blanks for indenting */
ARCharMenuItemStruct *value;      /* IN; value to print */

{
   char  buff[55];              /* temporary buffer */

   DriverPrintResult("%sChar Menu Item Struct:\n", indent);
   DriverPrintResult("%s", indent);
   PrintChar("   Label:", value->menuLabel);
   DriverPrintResult("%s", indent);
   PrintInt("   Type:", value->menuType);
   if (value->menuType == AR_MENU_TYPE_VALUE)
   {
      DriverPrintResult("%s", indent);
      PrintChar("   Value:", value->u.menuValue);
   }
   else if (value->menuType == AR_MENU_TYPE_MENU)
   {
      if (strlen(indent) < (size_t) 50)
      {
         (void) strcpy(buff, indent);
         (void) strcat(buff, "   ");
      }
      else  /* reset to 0 spaces */
         buff[0] = '\0';
      PrintARCharMenuStruct(buff, value->u.childMenu);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARCharMenuQueryStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuQueryStruct(indent, value)
char                  *indent;     /* IN; string of blanks for indenting */
ARCharMenuQueryStruct *value;      /* IN; value to print */

{
   DriverPrintResult("%sChar Menu Query Struct:\n", indent);
   DriverPrintResult("%s", indent);
   PrintChar("   Schema name:", value->schema);
   DriverPrintResult("%s", indent);
   PrintChar("   Server name:", value->server);
   PrintARQualifierStruct(indent, &value->qualifier);
   DriverPrintResult("%s", indent);
   PrintLong("   Label Field Id:", value->labelField);
   DriverPrintResult("%s", indent);
   PrintLong("   Value Field Id:", value->valueField);
   DriverPrintResult("%s", indent);
   PrintARBoolean("   Sort on Label:", value->sortOnLabel);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARCharMenuSQLStruct                          */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuSQLStruct(indent, value)
char                  *indent;     /* IN; string of blanks for indenting */
ARCharMenuSQLStruct   *value;      /* IN; value to print */

{
   DriverPrintResult("%sChar Menu SQL Struct:\n", indent);
   DriverPrintResult("%s", indent);
   PrintChar("   Server name:", value->server);
   DriverPrintResult("%s", indent);
   PrintChar("   SQL Command:", value->sqlCommand);
   DriverPrintResult("%s", indent);
   PrintLong("   Label Index:", value->labelIndex);
   DriverPrintResult("%s", indent);
   PrintLong("   Value Index:", value->valueIndex);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARCharMenuList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuList(indent, value)
char           *indent;       /* IN; string of blanks for indenting */
ARCharMenuList *value;        /* IN; value to print */

{
   int  i;                    /* working index */

   DriverPrintResult("%sCharacter Menu List : %u items\n", indent,
                     value->numItems);
   if (value->numItems != 0)
      for (i = 0; i < (int) value->numItems; i++)
         PrintARCharMenuItemStruct(indent, &value->charMenuList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARCharMenuStruct                           */
/*                                                                           */
/*****************************************************************************/

void PrintARCharMenuStruct(indent, value)
char             *indent;     /* IN; string of blanks for indenting */
ARCharMenuStruct *value;      /* IN; value to print */

{
   char  buff[55];            /* temporary buffer */

   if (strlen(indent) < (size_t) 50)
   {
      (void) strcpy(buff, indent);
      (void) strcat(buff, "   ");
   }
   else  /* reset to 0 spaces */
      buff[0] = '\0';

   DriverPrintResult("%sChar Menu Struct:\n", indent);
   DriverPrintResult("%sType of Menu:  ", buff);
   switch (value->menuType)
   {
      case AR_CHAR_MENU_NONE  :
         DriverPrintResult("NONE\n");
         break;
      case AR_CHAR_MENU_LIST  :
         DriverPrintResult("LIST\n");
         PrintARCharMenuList(buff, &value->u.menuList);
         break;
      case AR_CHAR_MENU_QUERY :
         DriverPrintResult("QUERY\n");
         PrintARCharMenuQueryStruct(buff, &value->u.menuQuery);
         break;
      case AR_CHAR_MENU_FILE  :
         DriverPrintResult("FILE\n");
         DriverPrintResult("%s   File Location:  ", buff);
         switch (value->u.menuFile.fileLocation)
         {
            case AR_MENU_FILE_SERVER :
               DriverPrintResult("SERVER\n");
               break;
            case AR_MENU_FILE_CLIENT :
               DriverPrintResult("CLIENT\n");
               break;
            default                   :
               DriverPrintResult("   <undefined file location = %u>\n",
                                 value->u.menuFile.fileLocation);
               break;
         }
         DriverPrintResult("%s   Filename:  %s\n", buff,
                           value->u.menuFile.filename);
         break;
      case AR_CHAR_MENU_SQL   :
         DriverPrintResult("SQL\n");
         PrintARCharMenuSQLStruct(buff, &value->u.menuSQL);
         break;
      default                   :
         DriverPrintResult("   <undefined menuType = %u>\n", value->menuType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARStructItemStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARStructItemStruct(value)
ARStructItemStruct *value;    /* IN; value to print */

{
   DriverPrintResult("   Structure Item Struct:  ");
   switch (value->type)
   {
      case AR_STRUCT_ITEM_SCHEMA      :
         DriverPrintResult("SCHEMA\n");
         break;
      case AR_STRUCT_ITEM_SCHEMA_DEFN :
         DriverPrintResult("SCHEMA DEFN\n");
         break;
      case AR_STRUCT_ITEM_SCHEMA_VIEW :
         DriverPrintResult("SCHEMA VIEW\n");
         break;
      case AR_STRUCT_ITEM_SCHEMA_MAIL :
         DriverPrintResult("SCHEMA MAIL\n");
         break;
      case AR_STRUCT_ITEM_FILTER      :
         DriverPrintResult("FILTER\n");
         break;
      case AR_STRUCT_ITEM_ACTIVE_LINK :
         DriverPrintResult("ACTIVE LINK\n");
         break;
      case AR_STRUCT_ITEM_ADMIN_EXT   :
         DriverPrintResult("ADMINISTRATOR EXTENSION\n");
         break;
      case AR_STRUCT_ITEM_CHAR_MENU   :
         DriverPrintResult("CHARACTER MENU\n");
         break;
      case AR_STRUCT_ITEM_ESCALATION   :
         DriverPrintResult("ESCALATION\n");
         break;
      case AR_STRUCT_ITEM_DIST_MAP     :
         DriverPrintResult("DISTRIBUTED MAPPING\n");
         break;
      default                         :
         DriverPrintResult("   <undefined type = %u>\n", value->type);
         break;
   }
   PrintChar("      Name:", value->name);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARStructItemList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARStructItemList(value)
ARStructItemList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARStructItemStruct *tempPtr;/* working pointer */

   DriverPrintResult("Structure Item List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->structItemList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARStructItemStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                        PrintARServerInfoRequestList                       */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoRequestList(header2, value)
char                    *header2;/* IN; header string for 2nd+ line */
ARServerInfoRequestList *value;  /* IN; value to print */

{
   char          buff[82];    /* buffer to build output in */
   char          buff2[16];   /* buffer to hold a converted number */
   int           i;           /* working index */
   unsigned int *tempPtr;     /* working pointer */

   (void) sprintf(buff, "Server Info Request List:  (%u)", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->requestList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (i != 0)
            (void) strcat(buff, ", ");
         (void) sprintf(buff2, "%u", *tempPtr);
         if ((strlen(buff) + strlen(buff2)) >= (size_t) 79)
         {
            buff[strlen(buff) - 1] = '\n'; /* over last blank */
            DriverPrintResult("%s", buff);
            (void) strcpy(buff, header2);
         }
         (void) strcat(buff, buff2);
         tempPtr++;
      }
   }
   DriverPrintResult("%s\n", buff);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARServerInfoStruct                         */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoStruct(value)
ARServerInfoStruct *value;    /* IN; value to print */

{
   DriverPrintResult("   Server Info Struct:  ");
   switch (value->operation)
   {
      case AR_SERVER_INFO_DB_TYPE        :
         DriverPrintResult("DB TYPE\n");
         break;
      case AR_SERVER_INFO_SERVER_LICENSE :
         DriverPrintResult("SERVER LICENSE\n");
         break;
      case AR_SERVER_INFO_FIXED_LICENSE  :
         DriverPrintResult("FIXED USER LICENSE\n");
         break;
      case AR_SERVER_INFO_VERSION        :
         DriverPrintResult("VERSION\n");
         break;
      case AR_SERVER_INFO_ALLOW_GUESTS   :
         DriverPrintResult("ALLOW GUESTS\n");
         break;
      case AR_SERVER_INFO_USE_ETC_PASSWD :
         DriverPrintResult("USE /etc/passwd\n");
         break;
      case AR_SERVER_INFO_XREF_PASSWORDS :
         DriverPrintResult("CROSSREF PASSWORDS\n");
         break;
      case AR_SERVER_INFO_DEBUG_MODE     :
         DriverPrintResult("DEBUG MODE\n");
         break;
      case AR_SERVER_INFO_DB_NAME        :
         DriverPrintResult("DB NAME\n");
         break;
      case AR_SERVER_INFO_DB_PASSWORD    :
         DriverPrintResult("DB PASSWORD\n");
         break;
      case AR_SERVER_INFO_HARDWARE       :
         DriverPrintResult("HARDWARE\n");
         break;
      case AR_SERVER_INFO_OS             :
         DriverPrintResult("OS\n");
         break;
      case AR_SERVER_INFO_SERVER_DIR     :
         DriverPrintResult("SERVER DIRECTORY\n");
         break;
      case AR_SERVER_INFO_DBHOME_DIR     :
         DriverPrintResult("DBHOME DIRECTORY\n");
         break;
      case AR_SERVER_INFO_SET_PROC_TIME  :
         DriverPrintResult("SET FILTER PROCESS TIMEOUT\n");
         break;
      case AR_SERVER_INFO_EMAIL_FROM     :
         DriverPrintResult("EMAIL FROM ADDRESS FOR NOTIFICATIONS\n");
         break;
      case AR_SERVER_INFO_SQL_LOG_FILE   :
         DriverPrintResult("SQL LOG FILENAME\n");
         break;
      case AR_SERVER_INFO_FLOAT_LICENSE  :
         DriverPrintResult("FLOATING USER LICENSE\n");
         break;
      case AR_SERVER_INFO_FLOAT_TIMEOUT  :
         DriverPrintResult("FLOATING LICENSE TIMEOUT\n");
         break;
      case AR_SERVER_INFO_UNQUAL_QUERIES :
         DriverPrintResult("ALLOW UNQUALIFIED QUERIES\n");
         break;
      case AR_SERVER_INFO_FILTER_LOG_FILE:
         DriverPrintResult("FILTER LOG FILENAME\n");
         break;
      case AR_SERVER_INFO_USER_LOG_FILE  :
         DriverPrintResult("USER LOG FILENAME\n");
         break;
      case AR_SERVER_INFO_REM_SERV_ID    :
         DriverPrintResult("REMEDY SERVER ID\n");
         break;
      case AR_SERVER_INFO_MULTI_SERVER   :
         DriverPrintResult("MULTI SERVER FLAG\n");
         break;
      case AR_SERVER_INFO_EMBEDDED_SQL   :
         DriverPrintResult("EMBEDDED SQL FLAG\n");
         break;
      case AR_SERVER_INFO_MAX_SCHEMAS    :
         DriverPrintResult("MAX SCHEMAS LICENSED\n");
         break;
      case AR_SERVER_INFO_DB_VERSION     :
         DriverPrintResult("DB VERSION\n");
         break;
      case AR_SERVER_INFO_MAX_ENTRIES    :
         DriverPrintResult("MAX ENTRIES PER QUERY\n");
         break;
      case AR_SERVER_INFO_MAX_F_DAEMONS  :
         DriverPrintResult("MAX FAST PROCESSES\n");
         break;
      case AR_SERVER_INFO_MAX_L_DAEMONS  :
         DriverPrintResult("MAX LIST PROCESSES\n");
         break;
      case AR_SERVER_INFO_ESCALATION_LOG_FILE:
         DriverPrintResult("ESCALATION LOG FILENAME\n");
         break;
      case AR_SERVER_INFO_ESCL_DAEMON    :
         DriverPrintResult("ESCALATION DAEMON\n");
         break;
      case AR_SERVER_INFO_SUBMITTER_MODE :
         DriverPrintResult("SUBMITTER MODE\n");
         break;
      case AR_SERVER_INFO_API_LOG_FILE   :
         DriverPrintResult("API LOG FILE\n");
         break;
      case AR_SERVER_INFO_FTEXT_FIXED    :
         DriverPrintResult("FULL TEXT FIXED LICENSES\n");
         break;
      case AR_SERVER_INFO_FTEXT_FLOAT    :
         DriverPrintResult("FULL TEXT FLOATING LICENSES\n");
         break;
      case AR_SERVER_INFO_FTEXT_TIMEOUT  :
         DriverPrintResult("FULL TEXT FLOATING LICENSE TIMEOUT\n");
         break;
      case AR_SERVER_INFO_RESERV1_A     :
         DriverPrintResult("RESERVED FIXED LICENSES\n");
         break;
      case AR_SERVER_INFO_RESERV1_B     :
         DriverPrintResult("RESERVED FLOATING LICENSES\n");
         break;
      case AR_SERVER_INFO_RESERV1_C     :
         DriverPrintResult("RESERVED BUILDER LICENSES\n");
         break;
      case AR_SERVER_INFO_SERVER_IDENT  :
         DriverPrintResult("UNIQUE SERVER IDENTIFIER\n");
         break;
      case AR_SERVER_INFO_DS_SVR_LICENSE :
         DriverPrintResult("DISTRIBUTED SERVER LICENSE\n");
         break;
      case AR_SERVER_INFO_DS_MAPPING :
         DriverPrintResult("DISTRIBUTED SERVER MAPPING SCHEMA\n");
         break;
      case AR_SERVER_INFO_DS_PENDING :
         DriverPrintResult("DISTRIBUTED SERVER PENDING SCHEMA\n");
         break;
      case AR_SERVER_INFO_DS_RPC_SOCKET :
         DriverPrintResult("DISTRIBUTED SERVER RPC SOCKET\n");
         break;
      case AR_SERVER_INFO_DS_LOG_FILE   :
         DriverPrintResult("DISTRIBUTED SERVER LOG FILE\n");
         break;
      case AR_SERVER_INFO_SUPPRESS_WARN :
         DriverPrintResult("SUPPRESSED WARNINGS\n");
         break;
      case AR_SERVER_INFO_HOSTNAME      :
         DriverPrintResult("SERVER HOSTNAME\n");
         break;
      case AR_SERVER_INFO_FULL_HOSTNAME :
         DriverPrintResult("SERVER FULL (DNS) HOSTNAME\n");
         break;
      case AR_SERVER_INFO_SAVE_LOGIN    :
         DriverPrintResult("SAVE LOGIN\n");
         break;
      case AR_SERVER_INFO_U_CACHE_CHANGE:
         DriverPrintResult("USER CACHE CHANGE TIME\n");
         break;
      case AR_SERVER_INFO_G_CACHE_CHANGE:
         DriverPrintResult("GROUP CACHE CHANGE TIME\n");
         break;
      case AR_SERVER_INFO_STRUCT_CHANGE :
         DriverPrintResult("STRUCTURE CHANGE TIME\n");
         break;
      case AR_SERVER_INFO_CASE_SENSITIVE:
         DriverPrintResult("CASE SENSITIVITY\n");
         break;
      case AR_SERVER_INFO_SERVER_LANG   :
         DriverPrintResult("SERVER LANGUAGE\n");
         break;
      case AR_SERVER_INFO_ADMIN_ONLY    :
         DriverPrintResult("ADMIN ONLY MODE\n");
         break;
      case AR_SERVER_INFO_CACHE_LOG_FILE :
         DriverPrintResult("SHARED CACHE LOG FILE\n");
         break;
      case AR_SERVER_INFO_FLASH_DAEMON   :
         DriverPrintResult("FLASHBOARDS DAEMON\n");
         break;
      case AR_SERVER_INFO_THREAD_LOG_FILE:
         DriverPrintResult("THREAD LOG FILE\n");
         break;
      case AR_SERVER_INFO_ADMIN_TCP_PORT :
         DriverPrintResult("ADMIN SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_ESCL_TCP_PORT  :
         DriverPrintResult("ESCALATION SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_FAST_TCP_PORT  :
         DriverPrintResult("FAST SERVERS TCP PORT (initial port)\n");
         break;
      case AR_SERVER_INFO_LIST_TCP_PORT  :
         DriverPrintResult("LIST SERVERS TCP PORT (initial port)\n");
         break;
      case AR_SERVER_INFO_FLASH_TCP_PORT :
         DriverPrintResult("FLASHBOARDS SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_TCD_TCP_PORT   :
         DriverPrintResult("TCD SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_DSO_DEST_PORT  :
         DriverPrintResult("DSO SERVER DESTINATION TCP PORT\n");
         break;
      case AR_SERVER_INFO_INFORMIX_DBN   :
         DriverPrintResult("INFORMIX DB NAME\n");
         break;
      case AR_SERVER_INFO_INFORMIX_TBC   :
         DriverPrintResult("INFORMIX ONCONFIG\n");
         break;
      case AR_SERVER_INFO_INGRES_VNODE   :
         DriverPrintResult("INGRES VNODE\n");
         break;
      case AR_SERVER_INFO_ORACLE_SID     :
         DriverPrintResult("ORACLE SID\n");
         break;
      case AR_SERVER_INFO_ORACLE_TWO_T   :
         DriverPrintResult("ORACLE TWO TASK\n");
         break;
      case AR_SERVER_INFO_SYBASE_CHARSET :
         DriverPrintResult("SYBASE CHARSET\n");
         break;
      case AR_SERVER_INFO_SYBASE_SERV    :
         DriverPrintResult("SYBASE DB SERVER NAME\n");
         break;
      case AR_SERVER_INFO_SHARED_MEM     :
         DriverPrintResult("SHARED MEMORY\n");
         break;
      case AR_SERVER_INFO_SHARED_CACHE   :
         DriverPrintResult("SHARED CACHE\n");
         break;
      case AR_SERVER_INFO_CACHE_SEG_SIZE :
         DriverPrintResult("SHARED CACHE SEGMENT SIZE\n");
         break;
      case AR_SERVER_INFO_DB_USER        :
         DriverPrintResult("DB USER\n");
         break;
      case AR_SERVER_INFO_NFY_TCP_PORT :
         DriverPrintResult("NOTIFIER SERVER TCP PORT\n");
         break;
      case AR_SERVER_INFO_FILT_MAX_TOTAL :
         DriverPrintResult("FILTER MAX TOTAL NUMBER\n");
         break;
      case AR_SERVER_INFO_FILT_MAX_STACK :
         DriverPrintResult("FILTER MAX STACK SIZE\n");
         break;
      case AR_SERVER_INFO_DEFAULT_ORDER_BY :
         DriverPrintResult("DEFAULT ORDER BY ENTRY ID\n");
         break;
      case AR_SERVER_INFO_DELAYED_CACHE :
         DriverPrintResult("DELAYED CACHE\n");
         break;
      case AR_SERVER_INFO_DSO_MERGE_STYLE :
         DriverPrintResult("DSO MERGE STYLE\n");
         break;
      case AR_SERVER_INFO_EMAIL_LINE_LEN :
         DriverPrintResult("EMAIL LINE LENGTH\n");
         break;
      case AR_SERVER_INFO_EMAIL_SYSTEM :
         DriverPrintResult("EMAIL SYSTEM\n");
         break;
      case AR_SERVER_INFO_INFORMIX_RELAY_MOD :
         DriverPrintResult("INFORMIX RELAY MODULE\n");
         break;
      case AR_SERVER_INFO_PS_RPC_SOCKET :
         DriverPrintResult("PRIVATE SERVER RPC SOCKETS / PORTS\n");
         break;
      case AR_SERVER_INFO_REGISTER_PORTMAPPER :
         DriverPrintResult("REGISTER PORTMAPPER\n");
         break;
      case AR_SERVER_INFO_SERVER_NAME :
         DriverPrintResult("SERVER NAME\n");
         break;
      case AR_SERVER_INFO_DBCONF :
         DriverPrintResult("DBCONF\n");
         break;
      case AR_SERVER_INFO_APPL_PENDING :
         DriverPrintResult("APPLICATION PENDING SCHEMA\n");
         break;
      case AR_SERVER_INFO_AP_RPC_SOCKET :
         DriverPrintResult("APPROVAL SERVER RPC SOCKET\n");
         break;
      case AR_SERVER_INFO_AP_LOG_FILE   :
         DriverPrintResult("APPROVAL SERVER LOG FILE\n");
         break;
      case AR_SERVER_INFO_AP_DEFN_CHECK :
         DriverPrintResult("APPLICATION DEFINITION CHECK INTERVAL\n");
         break;
      case AR_SERVER_INFO_MAX_LOG_FILE_SIZE:
         DriverPrintResult("MAXIMUM SIZE OF LOG FILES\n");
         break;
      case AR_SERVER_INFO_CLUSTERED_INDEX :
         DriverPrintResult("CLUSTERED INDEX\n");
         break;
      case AR_SERVER_INFO_ACTLINK_DIR :
         DriverPrintResult("ACTIVE LINK RUN PROCESS DIRECTORY\n");
         break;
      case AR_SERVER_INFO_ACTLINK_SHELL :
         DriverPrintResult("ACTIVE LINK RUN PROCESS SHELL\n");
         break;
      case AR_SERVER_INFO_USER_CACHE_UTILS :
         DriverPrintResult("ALLOW USER CACHE UTILITIES\n");
         break;
      case AR_SERVER_INFO_EMAIL_TIMEOUT :
         DriverPrintResult("EMAIL TIMEOUT\n");
         break;
      case AR_SERVER_INFO_EXPORT_VERSION :
         DriverPrintResult("SERVER EXPORT VERSION\n");
         break;
      case AR_SERVER_INFO_ENCRYPT_AL_SQL :
         DriverPrintResult("ENCRYPT AL DIRECT SQL\n");
         break;
      case AR_SERVER_INFO_SCC_ENABLED         :
         DriverPrintResult("SCC ENABLED\n");
         break;
      case AR_SERVER_INFO_SCC_PROVIDER_NAME   :
         DriverPrintResult("SCC PROVIDER NAME\n");
         break;
      case AR_SERVER_INFO_SCC_TARGET_DIR      :
         DriverPrintResult("SCC TARGET DIRECTORY\n");
         break;
      case AR_SERVER_INFO_SCC_COMMENT_CHECKIN :
         DriverPrintResult("SCC COMMENTS ON CHECKIN\n");
         break;
      case AR_SERVER_INFO_SCC_COMMENT_CHECKOUT:
         DriverPrintResult("SCC COMMENTS ON CHECKOUT\n");
         break;
      case AR_SERVER_INFO_SCC_INTEGRATION_MODE:
         DriverPrintResult("SCC INTEGRATION MODE\n");
         break;
      case AR_SERVER_INFO_EA_RPC_SOCKET    :
         DriverPrintResult("EXTERNAL AUTHENTICATION RPC SOCKET\n");
         break;
      case AR_SERVER_INFO_EA_RPC_TIMEOUT   :
         DriverPrintResult("EXTERNAL AUTHENTICATION RPC TIMEOUT\n");
         break;
      case AR_SERVER_INFO_USER_INFO_LISTS:
         DriverPrintResult("USER HASH LISTS\n");
         break;
      case AR_SERVER_INFO_USER_INST_TIMEOUT:
         DriverPrintResult("USER INSTANCE TIMEOUT\n");
         break;
      case AR_SERVER_INFO_DEBUG_GROUPID :
         DriverPrintResult("DEBUG FROM CLIENT GROUP ID\n");
         break;
      case AR_SERVER_INFO_APPLICATION_AUDIT:
         DriverPrintResult("APPLICATION AUDIT INFO\n");
         break;
      case AR_SERVER_INFO_EA_SYNC_TIMEOUT  :
         DriverPrintResult("EXTERNAL AUTHENTICATION SYNC TIMEOUT\n");
         break;
      case AR_SERVER_INFO_SERVER_TIME:
         DriverPrintResult("SERVER CURRENT TIME\n");
         break;
      case AR_SERVER_INFO_SVR_SEC_CACHE:
         DriverPrintResult("TIE SECURITY CACHE TO SERVER NAME\n");
         break;
      case AR_SERVER_INFO_LOGFILE_APPEND:
         DriverPrintResult("LOGFILE APPEND\n");
         break;
      case AR_SERVER_INFO_MINIMUM_API_VER :
         DriverPrintResult("MINIMUM API VERSION SUPPORTED\n");
         break;
      case AR_SERVER_INFO_MAX_AUDIT_LOG_FILE_SIZE :
         DriverPrintResult("MAXIMUM AUDIT LOG FILE SIZE\n");
         break;
      case AR_SERVER_INFO_CANCEL_QUERY:
         DriverPrintResult("CANCEL QUERY\n");
         break;
      default                            :
         DriverPrintResult("   <undefined operation = %u>\n", value->operation);
         break;
   }
   PrintARValueStruct("      ", &value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARServerInfoList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoList(value)
ARServerInfoList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARServerInfoStruct *tempPtr;/* working pointer */

   DriverPrintResult("Server Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->serverInfoList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARServerInfoStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARServerInfoStruct2                        */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoStruct2(value)
ARServerInfoStruct *value;    /* IN; value to print */

{
   DriverPrintResult("   Server Info Struct:  ");
   switch (value->operation)
   {
      case AR_SERVER_STAT_START_TIME     :
         DriverPrintResult("START TIME\n");
         break;
      case AR_SERVER_STAT_BAD_PASSWORD   :
         DriverPrintResult("NUMBER BAD PASSWORD LOGIN ATTEMPTS\n");
         break;
      case AR_SERVER_STAT_NO_WRITE_TOKEN :
         DriverPrintResult("FLOATING WRITE TOKEN UNAVAILABLE\n");
         break;
      case AR_SERVER_STAT_NO_FULL_TOKEN  :
         DriverPrintResult("FLOATING FULLTEXT TOKEN UNAVAILABLE\n");
         break;
      case AR_SERVER_STAT_CURRENT_USERS  :
         DriverPrintResult("NUMBER CURRENT USERS\n");
         break;
      case AR_SERVER_STAT_WRITE_FIXED    :
         DriverPrintResult("NUMBER CURRENT WRITE FIXED USERS\n");
         break;
      case AR_SERVER_STAT_WRITE_FLOATING :
         DriverPrintResult("NUMBER CURRENT WRITE FLOATING USERS\n");
         break;
      case AR_SERVER_STAT_WRITE_READ     :
         DriverPrintResult("NUMBER CURRENT WRITE READ-ONLY USERS\n");
         break;
      case AR_SERVER_STAT_FULL_FIXED     :
         DriverPrintResult("NUMBER CURRENT FULLTEXT FIXED USERS\n");
         break;
      case AR_SERVER_STAT_FULL_FLOATING  :
         DriverPrintResult("NUMBER CURRENT FULLTEXT FLOATING USERS\n");
         break;
      case AR_SERVER_STAT_FULL_NONE      :
         DriverPrintResult("NUMBER CURRENT FULLTEXT NONE USERS\n");
         break;
      case AR_SERVER_STAT_API_REQUESTS   :
         DriverPrintResult("NUMBER API REQUESTS\n");
         break;
      case AR_SERVER_STAT_API_TIME       :
         DriverPrintResult("API SECONDS\n");
         break;
      case AR_SERVER_STAT_ENTRY_TIME     :
         DriverPrintResult("ARxxxEntry SECONDS\n");
         break;
      case AR_SERVER_STAT_RESTRUCT_TIME  :
         DriverPrintResult("RESTRUCTURE SECONDS\n");
         break;
      case AR_SERVER_STAT_OTHER_TIME     :
         DriverPrintResult("OTHER SECONDS\n");
         break;
      case AR_SERVER_STAT_CACHE_TIME     :
         DriverPrintResult("CACHE SECONDS\n");
         break;
      case AR_SERVER_STAT_GET_E_COUNT    :
         DriverPrintResult("NUMBER GET ENTRY\n");
         break;
      case AR_SERVER_STAT_GET_E_TIME     :
         DriverPrintResult("GET ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_SET_E_COUNT    :
         DriverPrintResult("NUMBER SET ENTRY\n");
         break;
      case AR_SERVER_STAT_SET_E_TIME     :
         DriverPrintResult("SET ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_CREATE_E_COUNT :
         DriverPrintResult("NUMBER CREATE ENTRY\n");
         break;
      case AR_SERVER_STAT_CREATE_E_TIME  :
         DriverPrintResult("CREATE ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_DELETE_E_COUNT :
         DriverPrintResult("NUMBER DELETE ENTRY\n");
         break;
      case AR_SERVER_STAT_DELETE_E_TIME  :
         DriverPrintResult("DELETE ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_MERGE_E_COUNT  :
         DriverPrintResult("NUMBER MERGE ENTRY\n");
         break;
      case AR_SERVER_STAT_MERGE_E_TIME   :
         DriverPrintResult("MERGE ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_GETLIST_E_COUNT:
         DriverPrintResult("NUMBER GETLIST ENTRY\n");
         break;
      case AR_SERVER_STAT_GETLIST_E_TIME :
         DriverPrintResult("GETLIST ENTRY SECONDS\n");
         break;
      case AR_SERVER_STAT_E_STATS_COUNT  :
         DriverPrintResult("NUMBER ENTRY STATISTICS\n");
         break;
      case AR_SERVER_STAT_E_STATS_TIME   :
         DriverPrintResult("ENTRY STATISTICS SECONDS\n");
         break;
      case AR_SERVER_STAT_FILTER_PASSED  :
         DriverPrintResult("NUMBER FILTERS PASSED QUAL\n");
         break;
      case AR_SERVER_STAT_FILTER_FAILED  :
         DriverPrintResult("NUMBER FILTERS FAILED QUAL\n");
         break;
      case AR_SERVER_STAT_FILTER_DISABLE :
         DriverPrintResult("NUMBER FILTERS DISABLED\n");
         break;
      case AR_SERVER_STAT_FILTER_NOTIFY  :
         DriverPrintResult("NUMBER FILTER NOTIFY ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_MESSAGE :
         DriverPrintResult("NUMBER FILTER MESSAGE ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_LOG     :
         DriverPrintResult("NUMBER FILTER LOG ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_FIELDS  :
         DriverPrintResult("NUMBER FILTER FIELDS ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_PROCESS :
         DriverPrintResult("NUMBER FILTER PROCESS ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_TIME    :
         DriverPrintResult("FILTER SECONDS\n");
         break;
      case AR_SERVER_STAT_ESCL_PASSED    :
         DriverPrintResult("NUMBER ESCALATIONS PASSED QUAL\n");
         break;
      case AR_SERVER_STAT_ESCL_FAILED    :
         DriverPrintResult("NUMBER ESCALATIONS FAILED QUAL\n");
         break;
      case AR_SERVER_STAT_ESCL_DISABLE   :
         DriverPrintResult("NUMBER ESCALATIONS DISABLED\n");
         break;
      case AR_SERVER_STAT_ESCL_NOTIFY    :
         DriverPrintResult("NUMBER ESCALATION NOTIFY ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_LOG       :
         DriverPrintResult("NUMBER ESCALATION LOG ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_FIELDS    :
         DriverPrintResult("NUMBER ESCALATION FIELDS ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_PROCESS   :
         DriverPrintResult("NUMBER ESCALATION PROCESS ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_TIME      :
         DriverPrintResult("ESCALATION SECONDS\n");
         break;
      case AR_SERVER_STAT_TIMES_BLOCKED  :
         DriverPrintResult("TIMES AT LEAST ONE REQUEST IN QUEUE\n");
         break;
      case AR_SERVER_STAT_NUMBER_BLOCKED :
         DriverPrintResult("TOTAL NUMBER IN QUEUE DURING RUN\n");
         break;
      case AR_SERVER_STAT_CPU            :
         DriverPrintResult("CPU TIME FOR SERVER PROCESS\n");
         break;
      case AR_SERVER_STAT_SQL_DB_COUNT   :
         DriverPrintResult("NUMBER SQL DB OPERTIONS\n");
         break;
      case AR_SERVER_STAT_SQL_DB_TIME    :
         DriverPrintResult("SQL DB TIME\n");
         break;
      case AR_SERVER_STAT_FTS_SRCH_COUNT :
         DriverPrintResult("NUMBER FULLTEXT SEARCH OPERTIONS\n");
         break;
      case AR_SERVER_STAT_FTS_SRCH_TIME  :
         DriverPrintResult("SQL FULLTEXT SEARCH TIME\n");
         break;
      case AR_SERVER_STAT_SINCE_START    :
         DriverPrintResult("TIME SINCE START TIME\n");
         break;
      case AR_SERVER_STAT_IDLE_TIME      :
         DriverPrintResult("IDLE TIME BETWEEN CALLS\n");
         break;
      case AR_SERVER_STAT_NET_RESP_TIME  :
         DriverPrintResult("NETWORK TIME SHIPPING RESPONSES\n");
         break;
      case AR_SERVER_STAT_FILTER_FIELDP  :
         DriverPrintResult("NUMBER FILTER PUSH FIELDS ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_FIELDP    :
         DriverPrintResult("NUMBER ESCALATION PUSH FIELDS ACTIONS\n");
         break;
      case AR_SERVER_STAT_FILTER_SQL     :
         DriverPrintResult("NUMBER FILTER SQL ACTIONS\n");
         break;
      case AR_SERVER_STAT_ESCL_SQL       :
         DriverPrintResult("NUMBER ESCALATION SQL ACTIONS\n");
         break;
      case AR_SERVER_STAT_NUM_THREADS    :
         DriverPrintResult("NUMBER OF THREADS IN QUEUE\n");
         break;
      default                       :
         DriverPrintResult("   <undefined operation = %u>\n", value->operation);
         break;
   }
   PrintARValueStruct("      ", &value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARServerInfoList2                         */
/*                                                                           */
/*****************************************************************************/

void PrintARServerInfoList2(value)
ARServerInfoList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARServerInfoStruct *tempPtr;/* working pointer */

   DriverPrintResult("Server Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->serverInfoList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARServerInfoStruct2(tempPtr);
         tempPtr++;
      }
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintARFullTextInfoStruct                       */
/*                                                                           */
/*****************************************************************************/

void PrintARFullTextInfoStruct(value)
ARFullTextInfoStruct *value;    /* IN; value to print */

{
   DriverPrintResult("   FullText Info Struct:  ");
   switch (value->infoType)
   {
      case AR_FULLTEXTINFO_COLLECTION_DIR :
         DriverPrintResult("COLLECTION DIRECTORY\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      case AR_FULLTEXTINFO_STOPWORD:
         DriverPrintResult("STOPWORD LIST\n");
         PrintARValueList("      ", &value->u.valueList);
         break;
      case AR_FULLTEXTINFO_REINDEX:
         DriverPrintResult("COLLECTION REINDEX\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      case AR_FULLTEXTINFO_CASE_SENSITIVE_SRCH:
         DriverPrintResult("CASE INSENSITIVE SEARCH\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      case AR_FULLTEXTINFO_STATE:
         DriverPrintResult("FULLTEXT OPERATIONS STATE\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      case AR_FULLTEXTINFO_FTS_MATCH_OP:
         DriverPrintResult("FULLTEXT SEARCH MATCH OPERATOR\n");
         PrintARValueStruct("      ", &value->u.value);
         break;
      default                       :
         DriverPrintResult("   <undefined infoType = %u>\n", value->infoType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARFullTextInfoList                        */
/*                                                                           */
/*****************************************************************************/

void PrintARFullTextInfoList(value)
ARFullTextInfoList *value;      /* IN; value to print */

{
   int                 i;      /* working index */
   ARFullTextInfoStruct *tempPtr;/* working pointer */

   DriverPrintResult("FullText Info List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->fullTextInfoList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         PrintARFullTextInfoStruct(tempPtr);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARStatusHistory                          */
/*                                                                           */
/*****************************************************************************/

void PrintARStatusHistoryList(header, value)
char                *header;     /* IN; header string for the value */
ARStatusHistoryList *value;      /* IN; value to print */

{
   char                   buff[55];   /* temporary buffer */
   int                    i;          /* working index */
   int                    leadSpaces; /* number of lead spaces in header */
   ARStatusHistoryStruct *tempPtr;    /* working pointer */

   leadSpaces = 0;
   while (header[leadSpaces] == ' ')
      leadSpaces++;
   leadSpaces += 3;

   if (leadSpaces < (size_t) 50)
   {
      (void) memset(buff, ' ', leadSpaces);
      buff[leadSpaces] = '\0';
   }
   else  /* reset to 0 spaces */
      buff[0] = '\0';

   DriverPrintResult("%s %u stat hist items\n", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->statHistList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         if (tempPtr->user[0] == '\0')
            DriverPrintResult("%s %d - \n", buff, i);
         else
            DriverPrintResult("%s %d - %ld   %s\n", buff, i, tempPtr->timeVal,
                              tempPtr->user);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintARDiaryList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARDiaryList(header, value)
char             *header;     /* IN; header string for the value */
ARDiaryList      *value;      /* IN; value to print */

{
   char            buff[55];   /* temporary buffer */
   char            buff2[55];  /* temporary buffer */
   int             i;          /* working index */
   int             leadSpaces; /* number of lead spaces in header */
   ARDiaryStruct  *tempPtr;    /* working pointer */

   leadSpaces = 0;
   while (header[leadSpaces] == ' ')
      leadSpaces++;
   leadSpaces += 3;

   if (leadSpaces < (size_t) 50)
   {
      (void) memset(buff, ' ', leadSpaces);
      buff[leadSpaces] = '\0';
      (void) memset(buff2, ' ', (leadSpaces + 3));
      buff2[leadSpaces + 3] = '\0';
   }
   else  /* reset to 0 spaces */
   {
      buff[0] = '\0';
      (void) memset(buff2, ' ', 3);
      buff2[3] = '\0';
   }

   DriverPrintResult("%s %u diary entries\n", header, value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->diaryList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         DriverPrintResult("%s %ld   %s\n", buff, tempPtr->timeVal,
                           tempPtr->user);
         PrintChar(buff2, tempPtr->value);

         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARFieldMapping                             */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldMapping(header, fieldMap)
char                 *header;
ARFieldMappingStruct *fieldMap;               /* IN; value to print */

{
   DriverPrintResult("%s\n", header);
   DriverPrintResult("   Field type:  ");
   switch (fieldMap->fieldType)
   {
      case AR_FIELD_NONE :
         DriverPrintResult("<none>\n");
         break;
      case AR_FIELD_REGULAR :
         DriverPrintResult("REGULAR\n");
         break;
      case AR_FIELD_JOIN :
         DriverPrintResult("JOIN\n");
         PrintInt("  Schema Index: ", fieldMap->u.join.schemaIndex);
         PrintARInternalId("  Field Id :", fieldMap->u.join.realId);
         break;
      case AR_FIELD_VIEW :
         DriverPrintResult("VIEW\n");
         PrintChar("  External Field Name: ", fieldMap->u.view.fieldName);
         break;
      default             :
         DriverPrintResult("  <undefined fieldType = %u>\n",
                           fieldMap->fieldType);
         break;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARFieldMappingList                           */
/*                                                                           */
/*****************************************************************************/

void PrintARFieldMappingList(header, value)
char               *header;   /* IN; header string for the value */
ARFieldMappingList *value;    /* IN; value to print */

{
   int      i;                /* working index */

   DriverPrintResult("%s : %u items\n", header, value->numItems);
   for (i = 0; i < (int) value->numItems; i++)
      PrintARFieldMapping(" Field mapping: ", &value->mappingList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                            PrintARBooleanList                             */
/*                                                                           */
/*****************************************************************************/

void PrintARBooleanList(header, value)
char          *header;        /* IN; header string for the value */
ARBooleanList *value;         /* IN; value to print */

{
   unsigned int   i;                   /* working index */

   DriverPrintResult("%s : %u items\n", header, value->numItems);
   for (i = 0; i < value->numItems; i++)
   {
      if (value->booleanList[i])
         DriverPrintResult("   Item %3d : TRUE\n", i);
      else
         DriverPrintResult("   Item %3d : FALSE\n", i);
   }
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARContainerInfo                            */
/*                                                                           */
/*****************************************************************************/

void PrintARContainerInfo(header, value)
char                *header;        /* IN; header string for the value */
ARContainerInfo     *value;         /* IN; value to print */
{
   DriverPrintResult("%sContainer Info Name: %s\n", header, value->name);
   DriverPrintResult("%s   Type: %u\n", header, value->type);
   DriverPrintResult("%s   Owner Object Type: %u\n", header, value->ownerType);
   if (value->ownerType != ARCONOWNER_NONE &&
       value->ownerType != ARCONOWNER_ALL)
      DriverPrintResult("%s   Owner Object Name: %s\n", header,
                        value->ownerName);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARContainerInfoList                          */
/*                                                                           */
/*****************************************************************************/

void PrintARContainerInfoList(header, header2, value)
char                *header;        /* IN; header string for the value */
char                *header2;       /* IN; header string for 2nd+ line */
ARContainerInfoList *value;         /* IN; value to print */
{
   unsigned int   i;          /* working index */

   DriverPrintResult("%s (%u)\n", header, value->numItems);
   for (i = 0; i < value->numItems; i++)
      PrintARContainerInfo(header2, &value->conInfoList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                         PrintARContainerOwnerObj                          */
/*                                                                           */
/*****************************************************************************/
void PrintARContainerOwnerObj(value)
ARContainerOwnerObj *value;         /* IN; value to print */
{
   DriverPrintResult("Owner Object Type: %d\n", value->type);
   if (value->type != ARCONOWNER_NONE && value->type != ARCONOWNER_ALL)
      DriverPrintResult("Owner Object Name: %s\n", value->ownerName);
}


/*****************************************************************************/
/*                                                                           */
/*                             PrintARReference                              */
/*                                                                           */
/*****************************************************************************/

void PrintARReferenceStruct(header, value)
char                *header;        /* IN; header string for the value */
ARReferenceStruct   *value;         /* IN; value to print */
{
   char  buf[80];

   sprintf(buf, "%s   ", header);
   DriverPrintResult("%sReference Info:\n", header);
   DriverPrintResult("%s   Label: %s\n", header, value->label);
   DriverPrintResult("%s   Description: %s\n", header, value->description);
   DriverPrintResult("%s   Type: %u\n", header, value->type);
   DriverPrintResult("%s   Data Type: %u\n", header, value->reference.dataType);
   if (value->reference.dataType == ARREF_DATA_ARSREF)
   {
      DriverPrintResult("%s   ARS Reference: %s\n", header,
                        value->reference.u.name);
   }
   else if (value->reference.dataType == ARREF_DATA_EXTREF)
   {
      PrintARExtReferenceStruct(buf, &value->reference.u.extRef);
   }
}

/*****************************************************************************/
/*                                                                           */
/*                           PrintARExtReference                             */
/*                                                                           */
/*****************************************************************************/

void PrintARExtReferenceStruct(header, value)
char                 *header;        /* IN; header string for the value */
ARExtReferenceStruct *value;         /* IN; value to print */
{
   char  buf[80];

   DriverPrintResult("%sExternal Reference Info:\n", header);
   sprintf(buf, "%sGroup List:", header);
   PrintARInternalIdList(buf, header, &value->permittedGroups);
   PrintARValueStruct(header, &value->value);
}


/*****************************************************************************/
/*                                                                           */
/*                           PrintARReferenceList                            */
/*                                                                           */
/*****************************************************************************/

void PrintARReferenceList(value)
ARReferenceList *value;         /* IN; value to print */
{
   unsigned int   i;          /* working index */

   DriverPrintResult("Reference List (%u)\n", value->numItems);
   for (i = 0; i < value->numItems; i++)
      PrintARReferenceStruct("   ", &value->referenceList[i]);
}


/*****************************************************************************/
/*                                                                           */
/*                               PrintReturnCode                             */
/*                                                                           */
/*****************************************************************************/

void PrintReturnCode(returnCode)
int returnCode;               /* IN; value to print */

{
   DriverPrintResult("ReturnCode:  ");
   switch (returnCode)
   {
      case AR_RETURN_OK         :
         DriverPrintResult("OK\n");
         break;
      case AR_RETURN_WARNING    :
         DriverPrintResult("WARNING\n");
         break;
      case AR_RETURN_ERROR      :
         DriverPrintResult("ERROR\n");
         break;
      case AR_RETURN_FATAL      :
         DriverPrintResult("FATAL\n");
         break;
      case AR_RETURN_BAD_STATUS :
         DriverPrintResult("BAD STATUS\n");
         break;
      default                   :
         DriverPrintResult("<unknown return>\n");
         break;
   }
}
