/* File:  util.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef _WIN32
#include <windows.h>
#endif /* _WIN32 */

#include "ar.h"
#include "get.h"
#include "globals.h"
#include "main.h"
#include "util.h"


/*****************************************************************************/
/*                                                                           */
/*                                GetInputLine                               */
/*                                                                           */
/*****************************************************************************/

void GetInputLine()

{
   int                 bufLen; /* length of the buffer */
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */

   threadControlBlockPtr = GetThreadControlBlockPtr();

   if (fgets(threadControlBlockPtr->buffer, 1024,
             threadControlBlockPtr->inFile) == NULL)
      if (threadControlBlockPtr->inFile == stdin)
      {                          /* eof; print error and exit */
         (void) printf("\n*** EOF on input file; exiting\n");
         exit(1);
      }
      else
      {                          /* nested file; return to stdin */
         (void) fclose(threadControlBlockPtr->inFile);
         threadControlBlockPtr->inFile = stdin;
         GetInputLine();
         return;
      }
                                 /* record command if recording active */
   if (threadControlBlockPtr->recordFile != NULL)
      if ((strncmp(threadControlBlockPtr->buffer, commands[COMMAND_STOP_RECORD],
                   strlen(commands[COMMAND_STOP_RECORD])) != 0) ||
          (threadControlBlockPtr->buffer[
            strlen(commands[COMMAND_STOP_RECORD])] != '\n'))
         (void) fprintf(threadControlBlockPtr->recordFile, "%s",
                        threadControlBlockPtr->buffer);
                                 /* if output redirected AND comment, log */
                                 /*   comment to the output file          */
   if ((threadControlBlockPtr->buffer[0] == '#') &&
       (threadControlBlockPtr->outFile != stdout))
      (void) fprintf(threadControlBlockPtr->outFile, "%s",
                     threadControlBlockPtr->buffer);
                                 /* remove any trailing whitespace */
   bufLen = strlen(threadControlBlockPtr->buffer) - 1;
   while ((bufLen >= 0) && ((threadControlBlockPtr->buffer[bufLen] == ' ') ||
           (threadControlBlockPtr->buffer[bufLen] == '\t') ||
           (threadControlBlockPtr->buffer[bufLen] == '\n')))
      bufLen--;
   threadControlBlockPtr->buffer[bufLen + 1] = '\0';
}


/*****************************************************************************/
/*                                                                           */
/*                               OpenInputFile                               */
/*                                                                           */
/*****************************************************************************/

void OpenInputFile(inputFilename)
char     *inputFilename;
{
   char               *filename; /* pointer to filename for record file */
   FILE               *tempFile; /* temporary pointer to new input file */
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */

   /* NOTE: input files can contain calls to other input files, but when */
   /*    those nested files are completed, control DOES NOT return to    */
   /*    the calling file.  When EOF is encountered in an input file,    */
   /*    control is returned to the stdin file.  So, a file should       */
   /*    contain an ex command only as the last command in the file.     */
                                 /* get the filename of the new input file */
   if (inputFilename)
      filename = inputFilename;
   else
      filename = GetChar("Filename of input file (): ", "");
   if (strlen(filename) == 0)
      (void) printf(" **** No filename specified so no change to input file\n");
   else
   {                             /* open the new file for reading */
      tempFile = fopen(filename, "r");
      if (tempFile == NULL)
         (void) printf(
                     " **** File error during open; no change to input file\n");
      else
      {                          /* if there is a current input file; close it*/
         threadControlBlockPtr = GetThreadControlBlockPtr();
         if (threadControlBlockPtr->inFile != stdin)
            (void) fclose(threadControlBlockPtr->inFile);
         threadControlBlockPtr->inFile = tempFile;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               OpenOutputFile                              */
/*                                                                           */
/*****************************************************************************/

void OpenOutputFile()

{
   char               *filename; /* pointer to filename for record file */
   FILE               *tempFile; /* temporary pointer to new output file */
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */

                                 /* get the filename of the new output file */
   filename = GetChar("Filename of output file (): ", "");
   if (strlen(filename) == 0)
      (void)printf(" **** No filename specified so no change to output file\n");
   else
   {                             /* open the new file for writing */
      tempFile = fopen(filename, "w");
      if (tempFile == NULL)
         (void) printf(
                    " **** File error during open; no change to output file\n");
      else
      {                          /* if there is a current output file;close it*/
         threadControlBlockPtr = GetThreadControlBlockPtr();
         if (threadControlBlockPtr->outFile != stdout)
            (void) fclose(threadControlBlockPtr->outFile);
         threadControlBlockPtr->outFile = tempFile;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               CloseOutputFile                             */
/*                                                                           */
/*****************************************************************************/

void CloseOutputFile()

{
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */

   threadControlBlockPtr = GetThreadControlBlockPtr();

   if (threadControlBlockPtr->outFile == stdout)
      (void) printf(" **** Output to stdout; cannot close stdout file\n");
   else
   {                             /* close output file */
      (void) fclose(threadControlBlockPtr->outFile);
      threadControlBlockPtr->outFile = stdout;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                               StartRecording                              */
/*                                                                           */
/*****************************************************************************/

void StartRecording()

{
   char               *filename; /* pointer to filename for record file */
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */

   threadControlBlockPtr = GetThreadControlBlockPtr();

   if (threadControlBlockPtr->recordFile != NULL)
      (void) printf(
                " **** Recording already active; stop previous to start new\n");
   else
   {                             /* get a filename and open for recording */
      filename = GetChar("Filename of record file (): ", "");
      if (strlen(filename) == 0)
         (void) printf(" **** No filename specified so no recording started\n");
      else
      {                          /* open the new file for write */
         threadControlBlockPtr->recordFile = fopen(filename, "w");
         if (threadControlBlockPtr->recordFile == NULL)
            (void) printf(
                        " **** File error during open; no recording started\n");
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                                StopRecording                              */
/*                                                                           */
/*****************************************************************************/

void StopRecording()

{
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */

   threadControlBlockPtr = GetThreadControlBlockPtr();

   if (threadControlBlockPtr->recordFile == NULL)
      (void) printf(" **** Recording is not active\n");
   else
   {                             /* close record file */
      (void) fclose(threadControlBlockPtr->recordFile);
      threadControlBlockPtr->recordFile = NULL;
   }
}


/*****************************************************************************/
/*                                                                           */
/*                            ReleaseWaitingThreads                          */
/*                                                                           */
/*****************************************************************************/

void ReleaseWaitingThreads()

{
#ifndef SINGLE_THREADED
   ThreadControlBlock *threadControlBlockPtr; /* control block pointer */

   threadControlBlockPtr = GetThreadControlBlockPtr();

#ifdef _WIN32
   /* pause for a second to allow for all launched threads to be */
   /* waiting for the release event                              */
   
   Sleep(1000);

   /* release all threads waiting on this particular index */

   PulseEvent(threadControlBlockPtr->releaseWaitEvent);
#else
   /* pause for a second to allow for all launched threads to be */
   /* waiting for the release event                              */
   
   sleep(1);

   pthread_mutex_lock(&threadControlBlockPtr->releaseWaitMutex);
   pthread_cond_broadcast(&threadControlBlockPtr->releaseWaitCond);
   pthread_mutex_unlock(&threadControlBlockPtr->releaseWaitMutex);
#endif /* _WIN32 */
#else
   (void) printf(" **** Command not allowed in single threaded driver\n");
#endif /* !SINGLE_THREADED */
}


/*****************************************************************************/
/*                                                                           */
/*                                   Sleep                                   */
/*                                                                           */
/*****************************************************************************/

void SleepTimer()

{
   unsigned long  sleepSeconds; /* number of seconds to sleep */

   sleepSeconds = (unsigned long) GetLong("Number of seconds (0): ", "0");
#ifdef _WIN32
   Sleep(sleepSeconds * 1000);
#else
   sleep((unsigned) sleepSeconds);
#endif
}


/*****************************************************************************/
/*                                                                           */
/*                         CreateThreadControlBlock                          */
/*                                                                           */
/*****************************************************************************/

void * CreateThreadControlBlock()

{
   ThreadControlBlock *threadControlBlockPtr;

   if ((threadControlBlockPtr =
         (ThreadControlBlock *) malloc(sizeof(ThreadControlBlock))) == NULL)
   {
      (void) printf(" **** malloc error creating thread control block\n");
   }
   else
   {
      /* initialize the thread control block */

      memset((char *) threadControlBlockPtr, 0, sizeof(ThreadControlBlock));

      /* save the control block pointer in the appropriate location */

#ifdef SINGLE_THREADED
      gTCBPtr = threadControlBlockPtr;
#else
#ifdef _WIN32
      TlsSetValue(gTLSIndex, (void *) threadControlBlockPtr);

      if ((threadControlBlockPtr->launchWaitEvent =
            CreateEvent(NULL, TRUE, FALSE, NULL)) == NULL)
      {
         (void) printf(" **** unable to create an event\n");
      }

      if ((threadControlBlockPtr->releaseWaitEvent =
            CreateEvent(NULL, TRUE, FALSE, NULL)) == NULL)
      {
         (void) printf(" **** unable to create an event\n");
      }
#else
      pthread_setspecific(gTLSKey, (void *) threadControlBlockPtr);

#ifdef HPUX10
      pthread_mutex_init(&threadControlBlockPtr->launchWaitMutex,
                         pthread_mutexattr_default);
      pthread_cond_init(&threadControlBlockPtr->launchWaitCond,
                        pthread_condattr_default);
      pthread_mutex_init(&threadControlBlockPtr->releaseWaitMutex,
                         pthread_mutexattr_default);
      pthread_cond_init(&threadControlBlockPtr->releaseWaitCond,
                        pthread_condattr_default);
#else
      pthread_mutex_init(&threadControlBlockPtr->launchWaitMutex, NULL);
      pthread_cond_init(&threadControlBlockPtr->launchWaitCond, NULL);
      pthread_mutex_init(&threadControlBlockPtr->releaseWaitMutex, NULL);
      pthread_cond_init(&threadControlBlockPtr->releaseWaitCond, NULL);
#endif /* HPUX10 */
#endif /* _WIN32 */
#endif /* SINGLE_THREADED */
   }

   return ((void *) threadControlBlockPtr);
}


/*****************************************************************************/
/*                                                                           */
/*                         GetThreadControlBlockPtr                          */
/*                                                                           */
/*****************************************************************************/

void * GetThreadControlBlockPtr()

{
#ifndef SINGLE_THREADED
#ifdef HPUX10
   void *threadControlBlockPtr;
#endif /* HPUX10 */
#endif /* !SINGLE_THREADED */
   
   /* get the control block pointer from the appropriate location */

#ifdef SINGLE_THREADED
   return (gTCBPtr);
#else
#ifdef _WIN32
   return (TlsGetValue(gTLSIndex));
#else
#ifdef HPUX10
   pthread_getspecific(gTLSKey, &threadControlBlockPtr);
   return (threadControlBlockPtr);
#else
   return (pthread_getspecific(gTLSKey));
#endif /* HPUX10 */
#endif /* _WIN32 */
#endif /* SINGLE_THREADED */
}


/*****************************************************************************/
/*                                                                           */
/*                         DestroyThreadControlBlock                         */
/*                                                                           */
/*****************************************************************************/

void DestroyThreadControlBlock()

{
#ifndef SINGLE_THREADED
   unsigned int         i;
#endif /* !SINGLE_THREADED */
   ThreadControlBlock * threadControlBlockPtr;

   /* get the control block pointer for the current thread */

   threadControlBlockPtr = GetThreadControlBlockPtr();

   /* close any files that need closing */

   if (threadControlBlockPtr->inFile != stdin)
      (void) fclose(threadControlBlockPtr->inFile);

   if (threadControlBlockPtr->outFile != stdout)
      (void) fclose(threadControlBlockPtr->outFile);

   if (threadControlBlockPtr->recordFile != NULL)
      (void) fclose(threadControlBlockPtr->recordFile);

#ifndef SINGLE_THREADED
   /* wait until all threads launched by this thread have completed */

   if (threadControlBlockPtr->numHandles > 0)
   {
#ifdef _WIN32
      WaitForMultipleObjects(threadControlBlockPtr->numHandles,
                             threadControlBlockPtr->threadHandles,
                             TRUE, INFINITE);
#else
      for (i = 0; i < threadControlBlockPtr->numHandles; i++)
         pthread_join(threadControlBlockPtr->threadHandles[i], NULL);
#endif /* _WIN32 */
   }

   /* close the thread handles now that we know the threads have terminated */

   for (i = 0; i < threadControlBlockPtr->numHandles; i++)
   {
#ifdef _WIN32
      CloseHandle(threadControlBlockPtr->threadHandles[i]);
#else
#ifdef HPUX10
      pthread_detach(&threadControlBlockPtr->threadHandles[i]);
#else
      pthread_detach(threadControlBlockPtr->threadHandles[i]);
#endif /* HPUX10 */
#endif /* _WIN32 */
   }

   /* clean up the event handles */

#ifdef _WIN32
   if (threadControlBlockPtr->launchWaitEvent != NULL)
      CloseHandle(threadControlBlockPtr->launchWaitEvent);

   if (threadControlBlockPtr->releaseWaitEvent != NULL)
      CloseHandle(threadControlBlockPtr->releaseWaitEvent);
#else
   pthread_mutex_destroy(&threadControlBlockPtr->launchWaitMutex);
   pthread_cond_destroy(&threadControlBlockPtr->launchWaitCond);
   pthread_mutex_destroy(&threadControlBlockPtr->releaseWaitMutex);
   pthread_cond_destroy(&threadControlBlockPtr->releaseWaitCond);
#endif /* _WIN32 */
#endif /* !SINGLE_THREADED */

   free(threadControlBlockPtr);
}


/*****************************************************************************/
/*                                                                           */
/*                            GetControlStructPtr                            */
/*                                                                           */
/*****************************************************************************/

ARControlStruct * GetControlStructPtr()

{
   return(&((ThreadControlBlock *) GetThreadControlBlockPtr())->control);
}
