/* File: api.c */

#include <stdlib.h>
#ifdef _WIN32
#include <time.h>
#else /* _WIN32 */
#include <unistd.h>
#endif /* _WIN32 */
#include <stdio.h>
#include <string.h>

#include "ar.h"
#include "arextern.h"
#include "arfree.h"
#include "api.h"
#include "get.h"
#include "globals.h"
#include "print.h"
#include "util.h"


/*****************************************************************************/
/*                                                                           */
/*                           APIPrintStatusHistory                           */
/*                                                                           */
/*****************************************************************************/

void APIPrintStatusHistory(header, value)
char      *header;            /* IN; header string for the value */
char      *value;             /* IN; value to decode and print */

{
   ARStatusHistoryList  statHistList;
   int                  result;
   ARStatusList         status;

   /* Call routine to decode the value */
   result = ARDecodeStatusHistory(GetControlStructPtr(), value, &statHistList,
                                  &status);
   if (result >= AR_RETURN_ERROR)
   {
      DriverPrint("%s   DECODE STATUS-HISTORY FAILURE\n", header);
      PrintARStatusList(&status);
   }
   else
      PrintARStatusHistoryList(header, &statHistList);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARStatusHistoryList(&statHistList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIPrintDiary                                */
/*                                                                           */
/*****************************************************************************/

void APIPrintDiary(header, value)
char      *header;            /* IN; header string for the value */
char      *value;             /* IN; value to decode and print */

{
   ARDiaryList    diaryList;
   int            result;
   ARStatusList   status;

   /* Call routine to decode the value */
   result = ARDecodeDiary(GetControlStructPtr(), value, &diaryList, &status);
   if (result >= AR_RETURN_ERROR)
   {
      DriverPrint("%s   DECODE DIARY FAILURE\n", header);
      PrintARStatusList(&status);
   }
   else
      PrintARDiaryList(header, &diaryList);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARDiaryList(&diaryList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                           ENTRY                         **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                               APIARGetEntry                               */
/*                                                                           */
/*****************************************************************************/

void APIARGetEntry()

{
   AREntryIdList    entryId;
   ARFieldValueList fieldList;
   ARInternalIdList idList;
   int              result;
   int              numItems;
   ARNameType       schema;
   ARStatusList     status;

   /* Get Parameters */
   (void) printf("GET ENTRY\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   numItems = GetInt("Number of Items in EntryId ? (1): ", 1);
   entryId.numItems = numItems;
   entryId.entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
   if (entryId.entryIdList == NULL)
   {
      DriverPrint("**** malloc Failure in ARGetEntry call ****\n");
      return;
   }

   for (numItems=0; numItems < (int) entryId.numItems; numItems++)
      (void) strcpy(entryId.entryIdList[numItems], GetChar("Entry Id: ", ""));
   (void) printf("Ids of fields to retrieve:\n");
   GetARInternalIdList(&idList);

   /* Call routine */
   result = ARGetEntry(GetControlStructPtr(), schema, &entryId, &idList,
                       &fieldList, &status);

   /* Print results */
   DriverPrint("\n   ARGetEntry  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARFieldValueList(&fieldList);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARInternalIdList(&idList, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeARFieldValueList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
   FreeAREntryIdList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                       APIARGetMultipleEntries                             */
/*                                                                           */
/*****************************************************************************/

void APIARGetMultipleEntries()

{
   AREntryIdListList    entryId;
   ARBooleanList        existList;
   ARFieldValueListList fieldList;
   ARInternalIdList     idList;
   int                  result;
   int                  i;
   int                  numEntries;
   int                  numItems;
   ARNameType           schema;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("GET MULTIPLE ENTRIES\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   while ((numEntries = GetInt("Number of Entries in EntryId List ? (1): ", 1))
          <= 0)
      (void) printf("Number of entries must be greater than 0\n");
   entryId.numItems = numEntries;
   entryId.entryIdList = (AREntryIdList *) 
                         malloc(numEntries * sizeof(AREntryIdList));
   if (entryId.entryIdList == NULL)
   {
      DriverPrint("**** malloc Failure in ARGetMultipleEntries call ****\n");
      return;
   }
   for (i=0; i<numEntries; i++)
   {
      (void) printf("Enter entry id number %d\n", i+1);
      numItems = GetInt("Number of Items in EntryId ? (1): ", 1);

      entryId.entryIdList[i].numItems = numItems;
      entryId.entryIdList[i].entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
      if (entryId.entryIdList[i].entryIdList == NULL)
      {
         DriverPrint("**** malloc Failure in ARGetMultipleEntries call ****\n");
         return;
      }
      for (numItems=0; numItems < (int) entryId.entryIdList[i].numItems;
           numItems++)
         (void) strcpy(entryId.entryIdList[i].entryIdList[numItems],
                       GetChar("Entry Id: ", ""));
   }

   (void) printf("Ids of fields to retrieve:\n");
   GetARInternalIdList(&idList);

   /* Call routine */
   result = ARGetMultipleEntries(GetControlStructPtr(), schema, &entryId,
                                 &idList, &existList, &fieldList, &status);

   /* Print results */
   DriverPrint("\n   ARGetMultipleEntries  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARBooleanList("Entry exists list", &existList);
      PrintARFieldValueListList(&fieldList);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARInternalIdList(&idList, FALSE);
   FreeARBooleanList(&existList, FALSE);
   FreeARFieldValueListList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
   FreeAREntryIdListList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                               APIARSetEntry                               */
/*                                                                           */
/*****************************************************************************/

void APIARSetEntry()

{
   AREntryIdList    entryId;
   ARFieldValueList fieldList;
   ARTimestamp      getTime;
   int              result;
   int              numItems;
   ARNameType       schema;
   ARStatusList     status;
   int              option;

   /* Get Parameters */
   (void) printf("SET ENTRY\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   numItems = GetInt("Number of Items in EntryId ? (1): ", 1);
   entryId.numItems = numItems;
   entryId.entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
   if (entryId.entryIdList == NULL)
   {
      DriverPrint("**** malloc Failure in ARSetEntry call ****\n");
      return;
   }

   for (numItems=0; numItems < (int) entryId.numItems; numItems++)
      (void) strcpy(entryId.entryIdList[numItems], GetChar("Entry Id: ", ""));
   (void) printf("Field/value pairs to set:\n");
   GetARFieldValueList(&fieldList);
   getTime = GetARTimestamp("Time of Get operation (0): ", (ARTimestamp) 0);

   option = GetInt("SetEntry option ? (0): ", 0);
   /* Call routine */
   result = ARSetEntry(GetControlStructPtr(), schema, &entryId, &fieldList,
                       getTime, option, &status);

   /* Print results */
   DriverPrint("\n   ARSetEntry  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARFieldValueList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
   FreeAREntryIdList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateEntry                             */
/*                                                                           */
/*****************************************************************************/

void APIARCreateEntry()

{
   AREntryIdType    entryId;
   ARFieldValueList fieldList;
   int              result;
   ARNameType       schema;
   ARStatusList     status;

   /* Get Parameters */
   (void) printf("CREATE ENTRY\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   (void) printf("Field/value pairs to create:\n");
   GetARFieldValueList(&fieldList);

   /* Call routine */
   result = ARCreateEntry(GetControlStructPtr(), schema, &fieldList, entryId,
                          &status);

   /* Print results */
   DriverPrint("\n   ARCreateEntry  results\n");
   PrintReturnCode(result);
   PrintChar("Entry id: ", entryId);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARFieldValueList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteEntry                             */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteEntry()

{
   AREntryIdList    entryId;
   int              result;
   int              numItems;
   ARNameType       schema;
   ARStatusList     status;
   int              option;

   /* Get Parameters */
   (void) printf("DELETE ENTRY\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   numItems = GetInt("Number of Items in EntryId ? (1): ", 1);
   entryId.numItems = numItems;
   entryId.entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
   if (entryId.entryIdList == NULL)
   {
      DriverPrint("**** malloc Failure in ARDeleteEntry call ****\n");
      return;
   }

   for (numItems=0; numItems < (int) entryId.numItems; numItems++)
      (void) strcpy(entryId.entryIdList[numItems], GetChar("Entry Id: ", ""));

   option = GetInt("DeleteEntry option ? (0): ", 0);
   /* Call routine */
   result = ARDeleteEntry(GetControlStructPtr(), schema, &entryId, option,
                          &status);

   /* Print results */
   DriverPrint("\n   ARDeleteEntry  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
   FreeAREntryIdList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARMergeEntry                              */
/*                                                                           */
/*****************************************************************************/

void APIARMergeEntry()

{
   AREntryIdType    entryId;
   ARFieldValueList fieldList;
   unsigned int     mergeType;
   int              result;
   ARNameType       schema;
   ARStatusList     status;

   /* Get Parameters */
   (void) printf("MERGE ENTRY\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   (void) printf("Field/value pairs to merge:\n");
   GetARFieldValueList(&fieldList);
   mergeType = GetInt(
       "Operation on duplicate entry (error, new ID, overwrite, merge) (1-4): ",
       500);

   /* Call routine */
   result = ARMergeEntry(GetControlStructPtr(), schema, &fieldList, mergeType,
                         entryId, &status);

   /* Print results */
   DriverPrint("\n   ARMergeEntry  results\n");
   PrintReturnCode(result);
   PrintChar("Entry id: ", entryId);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARFieldValueList(&fieldList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListEntry                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetListEntry()

{
   AREntryListList      entryList;
   AREntryListFieldList getListFields;
   unsigned int         maxRetrieve;
   unsigned int         numMatches;
   unsigned int        *numMatchesPtr;
   ARQualifierStruct    qualifier;
   int                  result;
   ARSortList           sortList;
   ARNameType           schema;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("GETLIST ENTRY\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   GetARQualifierStruct(&qualifier);
   GetAREntryListFieldList(&getListFields);
   GetARSortList(&sortList);
   maxRetrieve = GetInt("Maximum number of entries to retrieve (500): ", 500);
   if (GetARBoolean("Get number of matches? (F): ", FALSE))
      numMatchesPtr = &numMatches;
   else
      numMatchesPtr = NULL;

   /* Call routine */
   result = ARGetListEntry(GetControlStructPtr(), schema, &qualifier,
                           &getListFields, &sortList, maxRetrieve, &entryList,
                           numMatchesPtr, &status);

   /* Print results */
   DriverPrint("\n   ARGetListEntry  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintAREntryListList(&entryList);
      if (numMatchesPtr != NULL)
          PrintInt("Number of matches: ", numMatches);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&qualifier, FALSE);
   FreeAREntryListFieldList(&getListFields, FALSE);
   FreeARSortList(&sortList, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeAREntryListList(&entryList, FALSE);
   FreeARStatusList(&status, FALSE);
}

/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListEntryWithFields                  */
/*                                                                           */
/*****************************************************************************/

void APIARGetListEntryWithFields()

{
   AREntryListFieldValueList      entryList;
   AREntryListFieldList getListFields;
   unsigned int         maxRetrieve;
   unsigned int         numMatches;
   unsigned int        *numMatchesPtr;
   ARQualifierStruct    qualifier;
   int                  result;
   ARSortList           sortList;
   ARNameType           schema;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("GETLIST ENTRY WITH FIELDS\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   GetARQualifierStruct(&qualifier);
   GetAREntryListFieldList(&getListFields);
   GetARSortList(&sortList);
   maxRetrieve = GetInt("Maximum number of entries to retrieve (500): ", 500);
   if (GetARBoolean("Get number of matches? (F): ", FALSE))
      numMatchesPtr = &numMatches;
   else
      numMatchesPtr = NULL;

   /* Call routine */
   result = ARGetListEntryWithFields(GetControlStructPtr(), schema, &qualifier,
                                     &getListFields, &sortList, maxRetrieve,
                                     &entryList, numMatchesPtr, &status);

   /* Print results */
   DriverPrint("\n   ARGetListEntryWithFields  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintAREntryListFieldValueList(&entryList);
      if (numMatchesPtr != NULL)
          PrintInt("Number of matches: ", numMatches);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&qualifier, FALSE);
   FreeAREntryListFieldList(&getListFields, FALSE);
   FreeARSortList(&sortList, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeAREntryListFieldValueList(&entryList, FALSE);
   FreeARStatusList(&status, FALSE);
}

/*****************************************************************************/
/*                                                                           */
/*                            APIARGetEntryStatistics                        */
/*                                                                           */
/*****************************************************************************/

void APIARGetEntryStatistics()

{
   ARInternalIdList          groupByList;
   ARQualifierStruct         qualifier;
   int                       result;
   ARStatisticsResultList    results;
   ARNameType                schema;
   ARStatusList              status;
   unsigned int              statistic;
   ARFieldValueOrArithStruct target;

   /* Get Parameters */
   (void) printf("GET ENTRY STATISTICS\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   GetARQualifierStruct(&qualifier);
   (void) printf("Target Expression:\n");
   GetARFieldValueOrArithStruct(&target);
   statistic = GetInt(
        "Statistic operation to perform (Count,Sum,Avg,Min,Max) (1-5)(1): ", 1);
   (void) printf("Ids of fields to group statistics by:\n");
   GetARInternalIdList(&groupByList);

   /* Call routine */
   result = ARGetEntryStatistics(GetControlStructPtr(), schema, &qualifier,
                                 &target, statistic, &groupByList, &results,
                                 &status);

   /* Print results */
   DriverPrint("\n   ARGetEntryStatistics  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARStatisticsResultList("", &results);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&qualifier, FALSE);
   FreeARFieldValueOrArithStruct(&target, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeARStatisticsResultList(&results, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetEntryBLOB                              */
/*                                                                           */
/*****************************************************************************/

void APIARGetEntryBLOB()

{
   ARInternalId              fieldId;
   AREntryIdList             entryId;
   int                       result;
   int                       numItems;
   ARNameType                schema;
   ARStatusList              status;
   ARLocStruct               loc;

   /* Get Parameters */
   (void) printf("GET ENTRY BLOB\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   numItems = GetInt("Number of Items in EntryId ? (1): ", 1);
   entryId.numItems = numItems;
   entryId.entryIdList = (AREntryIdType *) 
                         malloc(numItems * sizeof(AREntryIdType));
   if (entryId.entryIdList == NULL)
   {
      DriverPrint("**** malloc Failure in ARGetEntry call ****\n");
      return;
   }

   for (numItems=0; numItems < (int) entryId.numItems; numItems++)
      (void) strcpy(entryId.entryIdList[numItems], GetChar("Entry Id: ", ""));
   fieldId = GetLong("Field Id: ", 1L);
   GetARLocStructForGet(&loc);

   /* Call routine */
   result = ARGetEntryBLOB(GetControlStructPtr(), schema, &entryId, 
                           fieldId, &loc, &status);

   /* Print results */
   DriverPrint("\n   ARGetEntryBLOB  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
   FreeARLocStruct(&loc, FALSE);
   FreeAREntryIdList(&entryId, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARLoadARQualifierStruct                      */
/*                                                                           */
/*****************************************************************************/

void APIARLoadARQualifierStruct()

{
   ARNameType        displayTag;
   ARQualifierStruct qualifier;
   char              qualString[1024];
   int               result;
   ARNameType        schema;
   ARStatusList      status;

   /* Get Parameters */
   (void) printf("LOAD ARQualifierStruct\n");
   (void) strcpy(schema, GetChar("Schema: ", ""));
   (void) strcpy(displayTag, GetChar("Display Tag (): ", ""));
   (void) strcpy(qualString, GetChar("Qualification string: ", ""));

   /* Call routine */
   result = ARLoadARQualifierStruct(GetControlStructPtr(), schema, displayTag,
                                    qualString, &qualifier, &status);

   /* Print results */
   DriverPrint("\n   ARLoadARQualifierStruct  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARQualifierStruct("", &qualifier);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&qualifier, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                          STRUCTURE                      **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                              APIARGetContainer                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetContainer()
{
   ARNameType           containerName;
   ARReferenceTypeList  refTypes;
   ARPermissionList     groupList;
   ARInternalIdList     admingrpList;
   ARContainerOwnerObj  ownerObj;
   char                *label;
   char                *description;
   unsigned int         type;
   ARReferenceList      references;
   char                *helpText;
   ARNameType           owner;
   ARTimestamp          timestamp;
   ARNameType           lastChanged;
   char                *changeDiary;
   ARStatusList         status;
   int                  result;

   /* Get Parameters */
   printf("GET CONTAINER\n");
   strcpy(containerName, GetChar("Container name: ", ""));
   GetARReferenceTypeList(&refTypes);

   /* Call routine */
   result = ARGetContainer(GetControlStructPtr(), containerName, &refTypes, &groupList,
                           &admingrpList, &ownerObj, &label, &description, &type,
                           &references, &helpText, owner, &timestamp, lastChanged,
                           &changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARGetContainer  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARPermissionList(&groupList);
      PrintARInternalIdList("Sub Admin Group list: ", "   ",
                            &admingrpList);
      PrintARContainerOwnerObj(&ownerObj);
      PrintChar("Label: ", label);
      PrintChar("Description: ", description);
      PrintInt("Type:", type);
      PrintARReferenceList(&references);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARPermissionList(&groupList, FALSE);
      FreeARInternalIdList(&admingrpList, FALSE);
      FreeARReferenceList(&references, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARSetContainer                            */
/*                                                                           */
/*****************************************************************************/

void APIARSetContainer()
{
   ARNameType            containerName;
   ARNameType            newName;
   char                 *newNamePtr;
   ARPermissionList      groupList;
   ARPermissionList     *groupListPtr;
   ARInternalIdList      admingrpList;
   ARInternalIdList     *admingrpListPtr;
   ARContainerOwnerObj   ownerObj;
   ARContainerOwnerObj  *ownerObjPtr;
   char                  label[256];
   char                 *labelPtr;
   char                  description[ARMAX_CON_DESCRIPTION_LEN + 1];
   char                 *descriptionPtr;
   unsigned int          type;
   unsigned int         *typePtr;
   ARReferenceList       references;
   ARReferenceList      *referencesPtr;
   ARBoolean             removeFlag;
   char                  helpText[1024];
   char                 *helpTextPtr;
   ARNameType            owner;
   char                 *ownerPtr;
   char                  changeDiary[1024];
   char                 *changeDiaryPtr;
   ARStatusList          status;
   int                   result;

   /* Get Parameters */
   (void) printf("SET CONTAINER\n");
   (void) strcpy(containerName, GetChar("Container name: ", ""));
   if (GetARBoolean("Change container name? (F): ", FALSE))
   {
      (void) strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change group list? (F): ", FALSE))
   {
      GetARPermissionList(FALSE, &groupList);
      groupListPtr = &groupList;
   }
   else
      groupListPtr = NULL;
   if (GetARBoolean("Change sub admin group list? (F): ", FALSE))
   {
      (void) printf("Ids of sub admin groups allowed to access container:\n");
      GetARInternalIdList(&admingrpList);
      admingrpListPtr = &admingrpList;
   }
   else
      admingrpListPtr = NULL;
   if (GetARBoolean("Change container owner object? (F): ", FALSE))
   {
      GetARContainerOwnerObj(&ownerObj);
      ownerObjPtr = &ownerObj;
   }
   else
      ownerObjPtr = NULL;
   if (GetARBoolean("Change label? (F): ", FALSE))
   {
      (void) strcpy(label, GetChar("   Label (): ", ""));
      labelPtr = label;
   }
   else
      labelPtr = NULL;
   if (GetARBoolean("Change description? (F): ", FALSE))
   {
      (void) strcpy(description, GetChar("   Description (): ", ""));
      descriptionPtr = description;
   }
   else
      descriptionPtr = NULL;
   if (GetARBoolean("Change container type? (F): ", FALSE))
   {
      type = GetInt("Guide, Application (1-2) (2): ", 2);
      typePtr = &type;
   }
   else
      typePtr = NULL;
   if (GetARBoolean("Change reference list? (F): ", FALSE))
   {
      GetARReferenceList(&references);
      referencesPtr = &references;
      removeFlag = GetARBoolean("Remove invalid references? (F): ", FALSE);
   }
   else
      referencesPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      (void) strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      (void) strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      (void) strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   result = ARSetContainer(GetControlStructPtr(), containerName, newNamePtr, groupListPtr,
                           admingrpListPtr, ownerObjPtr, labelPtr, descriptionPtr,
                           typePtr, referencesPtr, removeFlag, helpTextPtr, ownerPtr,
                           changeDiaryPtr, &status);

   /* Print results */
   DriverPrint("\n   ARSetContainer  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (groupListPtr != NULL)
      FreeARPermissionList(&groupList, FALSE);
   if (admingrpListPtr != NULL)
      FreeARInternalIdList(&admingrpList, FALSE);
   if (referencesPtr != NULL)
      FreeARReferenceList(&references, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARCreateContainer                           */
/*                                                                           */
/*****************************************************************************/

void APIARCreateContainer()
{
   ARNameType            containerName;
   ARPermissionList      groupList;
   ARInternalIdList      admingrpList;
   ARContainerOwnerObj   ownerObj;
   char                  label[256];
   char                  description[ARMAX_CON_DESCRIPTION_LEN + 1];
   unsigned int          type;
   ARReferenceList       references;
   ARBoolean             removeFlag;
   char                  helpText[1024];
   ARNameType            owner;
   char                  changeDiary[1024];
   ARStatusList          status;
   int                   result;

   /* Get Parameters */
   printf("CREATE CONTAINER\n");
   strcpy(containerName, GetChar("Container name: ", ""));
   GetARPermissionList(FALSE, &groupList);
   printf("Ids of sub admin groups allowed to access container:\n");
   GetARInternalIdList(&admingrpList);
   GetARContainerOwnerObj(&ownerObj);
   strcpy(label, GetChar("   Label (): ", ""));
   strcpy(description, GetChar("   Description (): ", ""));
   type = GetInt("Guide, Application (1-2) (2): ", 2);
   GetARReferenceList(&references);
   removeFlag = GetARBoolean("Remove invalid references? (F): ", FALSE);
   strcpy(helpText, GetChar("Help Text (): ", ""));
   strcpy(owner, GetChar("Owner (): ", ""));
   strcpy(changeDiary, GetChar("Change Diary (): ", ""));

   /* Call routine */
   result = ARCreateContainer(GetControlStructPtr(), containerName, &groupList,
                              &admingrpList, &ownerObj, label, description,
                              type, &references, removeFlag, helpText, owner,
                              changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARCreateContainer  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARPermissionList(&groupList, FALSE);
   FreeARInternalIdList(&admingrpList, FALSE);
   FreeARReferenceList(&references, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARDeleteContainer                           */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteContainer()
{
   ARNameType       containerName;
   ARStatusList     status;
   int              result;

   /* Get Parameters */
   (void) printf("DELETE CONTAINER\n");
   (void) strcpy(containerName, GetChar("Container name: ", ""));

   /* Call routine */
   result = ARDeleteContainer(GetControlStructPtr(), containerName, &status);

   /* Print results */
   DriverPrint("\n   ARDeleteContainer  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetListContainer                          */
/*                                                                           */
/*****************************************************************************/

void APIARGetListContainer()
{
   ARTimestamp           changedSince;
   ARContainerTypeList   containerTypes;
   unsigned int          attributes;
   ARContainerOwnerObj   ownerObj;
   ARContainerInfoList   conList;
   ARStatusList          status;
   int                   result;

   /* Get Parameters */
   (void) printf("GETLIST CONTAINER\n");
   changedSince = GetARTimestamp("Get all changed since (0): ", (ARTimestamp)0);
   GetARContainerTypeList(&containerTypes);
   attributes = GetInt("Container attributes (none, hidden) (0-1) (0): ", 0);
   GetARContainerOwnerObj(&ownerObj);

   /* Call routine */
   result = ARGetListContainer(GetControlStructPtr(), changedSince, &containerTypes,
                               attributes, &ownerObj, &conList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListContainer  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARContainerInfoList("Container list: ", "   ", &conList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARContainerInfoList(&conList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARGetSchema                             */
/*                                                                           */
/*****************************************************************************/

void APIARGetSchema()

{
   char                *changeDiary;
   AREntryListFieldList getListFields;
   ARPermissionList     groupList;
   ARInternalIdList     admingrpList;
   char                *helpText;
   ARIndexList          indexList;
   ARCompoundSchema     comSchema;
   ARNameType           lastChanged;
   ARNameType           owner;
   int                  result;
   ARNameType           schemaName;
   ARSortList           sortList;
   ARStatusList         status;
   ARTimestamp          timestamp;

   /* Get Parameters */
   (void) printf("GET SCHEMA\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));

   /* Call routine */
   result = ARGetSchema(GetControlStructPtr(), schemaName, &comSchema,
                        &groupList, &admingrpList, &getListFields, &sortList,
                        &indexList, &helpText, &timestamp, owner, lastChanged,
                        &changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARGetSchema  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARCompoundSchema(&comSchema);
      PrintARPermissionList(&groupList, FALSE);
      PrintARInternalIdList("Sub Admin Group list: ", "   ", &admingrpList);
      PrintAREntryListFieldList(&getListFields);
      PrintARSortList(&sortList);
      PrintARIndexList(&indexList);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARCompoundSchema(&comSchema, FALSE);
      FreeARPermissionList(&groupList, FALSE);
      FreeARInternalIdList(&admingrpList, FALSE);
      FreeAREntryListFieldList(&getListFields, FALSE);
      FreeARSortList(&sortList, FALSE);
      FreeARIndexList(&indexList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARSetSchema                             */
/*                                                                           */
/*****************************************************************************/

void APIARSetSchema()

{
   char                  changeDiary[1024];
   char                 *changeDiaryPtr;
   AREntryListFieldList  getListFields;
   AREntryListFieldList *getListFieldsPtr;
   ARPermissionList      groupList;
   ARPermissionList     *groupListPtr;
   ARInternalIdList      admingrpList;
   ARInternalIdList     *admingrpListPtr;
   char                  helpText[1024];
   char                 *helpTextPtr;
   ARIndexList           indexList;
   ARIndexList          *indexListPtr;
   ARCompoundSchema      comSchema;
   ARCompoundSchema     *comSchemaPtr;
   ARNameType            newName;
   char                 *newNamePtr;
   ARNameType            owner;
   char                 *ownerPtr;
   int                   result;
   ARNameType            schemaName;
   ARSortList            sortList;
   ARSortList           *sortListPtr;
   ARStatusList          status;

   /* Get Parameters */
   (void) printf("SET SCHEMA\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   if (GetARBoolean("Change schema name? (F): ", FALSE))
   {
      (void) strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change compound schema info? (F): ", FALSE))
   {
      GetARCompoundSchema(&comSchema);
      comSchemaPtr = &comSchema;
   }
   else
      comSchemaPtr = NULL;
   if (GetARBoolean("Change group list? (F): ", FALSE))
   {
      GetARPermissionList(FALSE, &groupList);
      groupListPtr = &groupList;
   }
   else
      groupListPtr = NULL;
   if (GetARBoolean("Change sub admin group list? (F): ", FALSE))
   {
      (void) printf("Ids of sub admin groups allowed to access schema:\n");
      GetARInternalIdList(&admingrpList);
      admingrpListPtr = &admingrpList;
   }
   else
      admingrpListPtr = NULL;
   if (GetARBoolean("Change query list fields? (F): ", FALSE))
   {
      GetAREntryListFieldList(&getListFields);
      getListFieldsPtr = &getListFields;
   }
   else
      getListFieldsPtr = NULL;
   if (GetARBoolean("Change sort list? (F): ", FALSE))
   {
      GetARSortList(&sortList);
      sortListPtr = &sortList;
   }
   else
      sortListPtr = NULL;
   if (GetARBoolean("Change index list? (F): ", FALSE))
   {
      GetARIndexList(&indexList);
      indexListPtr = &indexList;
   }
   else
      indexListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      (void) strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      (void) strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      (void) strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   result = ARSetSchema(GetControlStructPtr(), schemaName, newNamePtr,
                        comSchemaPtr, groupListPtr, admingrpListPtr,
                        getListFieldsPtr, sortListPtr, indexListPtr,
                        helpTextPtr, ownerPtr, changeDiaryPtr, &status);

   /* Print results */
   DriverPrint("\n   ARSetSchema  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (comSchemaPtr != NULL)
      FreeARCompoundSchema(&comSchema, FALSE);
   if (groupListPtr != NULL)
      FreeARPermissionList(&groupList, FALSE);
   if (admingrpListPtr != NULL)
      FreeARInternalIdList(&admingrpList, FALSE);
   if (getListFieldsPtr != NULL)
      FreeAREntryListFieldList(&getListFields, FALSE);
   if (sortListPtr != NULL)
      FreeARSortList(&sortList, FALSE);
   if (indexListPtr != NULL)
      FreeARIndexList(&indexList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateSchema                            */
/*                                                                           */
/*****************************************************************************/

void APIARCreateSchema()

{
   char                 changeDiary[1024];
   AREntryListFieldList getListFields;
   ARPermissionList     groupList;
   ARInternalIdList     admingrpList;
   char                 helpText[1024];
   ARIndexList          indexList;
   ARCompoundSchema     comSchema;
   ARNameType           owner;
   int                  result;
   ARNameType           schemaName;
   ARSortList           sortList;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("CREATE SCHEMA\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   GetARCompoundSchema(&comSchema);
   GetARPermissionList(FALSE, &groupList);
   (void) printf("Ids of sub admin groups allowed to access schema:\n");
   GetARInternalIdList(&admingrpList);
   GetAREntryListFieldList(&getListFields);
   GetARSortList(&sortList);
   GetARIndexList(&indexList);
   (void) strcpy(helpText, GetChar("Help Text (): ", ""));
   (void) strcpy(owner, GetChar("Owner (): ", ""));
   (void) strcpy(changeDiary, GetChar("Change Diary (): ", ""));

   /* Call routine */
   result = ARCreateSchema(GetControlStructPtr(), schemaName, &comSchema,
                           &groupList, &admingrpList, &getListFields, &sortList,
                           &indexList, helpText, owner, changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARCreateSchema  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARCompoundSchema(&comSchema, FALSE);
   FreeARPermissionList(&groupList, FALSE);
   FreeARInternalIdList(&admingrpList, FALSE);
   FreeAREntryListFieldList(&getListFields, FALSE);
   FreeARSortList(&sortList, FALSE);
   FreeARIndexList(&indexList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteSchema                            */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteSchema()

{
   unsigned int     deleteOption;
   int              result;
   ARNameType       schemaName;
   ARStatusList     status;

   /* Get Parameters */
   (void) printf("DELETE SCHEMA\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   deleteOption = GetInt("Delete option ? (0): ", 0);

   /* Call routine */
   result = ARDeleteSchema(GetControlStructPtr(), schemaName, deleteOption,
                           &status);

   /* Print results */
   DriverPrint("\n   ARDeleteSchema  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListSchema                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetListSchema()

{
   ARTimestamp      changedSince;
   int              result;
   ARNameList       schemaList;
   int              schemaType;
   ARNameType       joinName;
   ARStatusList     status;

   /* Get Parameters */
   (void) printf("GETLIST SCHEMA\n");
   changedSince = GetARTimestamp("Get all changed since (0): ", (ARTimestamp)0);
   schemaType = GetInt(
       "Type of schema (all,regular,join,view,uplink,downlink,dialog) (0-6) (0): ", 0);
   if (GetARBoolean("Include hidden schemas (T): ", TRUE))
      schemaType += AR_HIDDEN_INCREMENT;
   if (((schemaType % AR_HIDDEN_INCREMENT) == AR_LIST_SCHEMA_UPLINK) ||
       ((schemaType % AR_HIDDEN_INCREMENT) == AR_LIST_SCHEMA_DOWNLINK))
      (void) strcpy(joinName, GetChar("Schema name: ", ""));
   else
      joinName[0] = '\0';

   /* Call routine */
   result = ARGetListSchema(GetControlStructPtr(), changedSince, schemaType,
                            joinName, &schemaList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListSchema  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Schema list: ", "   ", &schemaList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&schemaList, FALSE);
   FreeARStatusList(&status, FALSE);
}

/*****************************************************************************/
/*                                                                           */
/*                                APIARGetField                              */
/*                                                                           */
/*****************************************************************************/

void APIARGetField()

{
   char              *changeDiary;
   unsigned int       createMode;
   unsigned int       dataType;
   ARValueStruct      defaultVal;
   ARDisplayInstanceList dInstanceList;
   ARInternalId       fieldId;
   ARNameType         fieldName;
   char              *helpText;
   ARNameType         lastChanged;
   ARFieldLimitStruct limit;
   ARPermissionList   permissions;
   unsigned int       option;
   ARNameType         owner;
   int                result;
   ARNameType         schemaName;
   ARStatusList       status;
   ARTimestamp        timestamp;
   ARFieldMappingStruct fieldMap;

   /* Get Parameters */
   (void) printf("GET FIELD\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   fieldId = GetLong("Field id (1): ", 1);

   /* Call routine */
   result = ARGetField(GetControlStructPtr(), schemaName, fieldId, fieldName,
                       &fieldMap, &dataType, &option, &createMode, &defaultVal,
                       &permissions, &limit, &dInstanceList, &helpText,
                       &timestamp, owner,lastChanged, &changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARGetField  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintChar("Field name: ", fieldName);
      PrintARFieldMapping("Field mapping: ", &fieldMap);
      PrintInt("Datatype: ", dataType);
      PrintInt("Field option: ", option);
      PrintInt("Field create mode: ", createMode);
      PrintARValueStruct("", &defaultVal);
      PrintARPermissionList(&permissions, TRUE);
      PrintARFieldLimitStruct(&limit);
      PrintARDisplayInstanceList(&dInstanceList);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARValueStruct(&defaultVal, FALSE);
      FreeARPermissionList(&permissions, FALSE);
      FreeARFieldLimitStruct(&limit, FALSE);
      FreeARDisplayInstanceList(&dInstanceList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARSetField                              */
/*                                                                           */
/*****************************************************************************/

void APIARSetField()

{
   char                changeDiary[1024];
   char               *changeDiaryPtr;
   unsigned int        createMode;
   unsigned int       *createModePtr;
   unsigned int        dataType;
   ARValueStruct       defaultVal;
   ARValueStruct      *defaultValPtr;
   ARDisplayInstanceList dInstanceList;
   ARDisplayInstanceList *dInstanceListPtr;
   ARInternalId        fieldId;
   ARNameType          fieldName;
   char               *fieldNamePtr;
   char                helpText[1024];
   char               *helpTextPtr;
   ARFieldLimitStruct  limit;
   ARFieldLimitStruct *limitPtr;
   unsigned int        option;
   unsigned int       *optionPtr;
   ARNameType          owner;
   char               *ownerPtr;
   ARPermissionList    permissions;
   ARPermissionList   *permissionsPtr;
   int                 result;
   ARNameType          schemaName;
   ARStatusList        status;

   /* Get Parameters */
   (void) printf("SET FIELD\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   fieldId = GetLong("Field id (1): ", 1);
   dataType = GetInt(
"Datatype Null/Key/Int/Real/Char/Diary/Enum/Time/Bitmask/Byte/Decimal/\n         attach/join/trim/control/Table/Column/ulong/coords\n         (0 - 11, 30-34, 40-41) (2): ",
                     AR_DATA_TYPE_INTEGER);
   if (GetARBoolean("Change fieldName? (F): ", FALSE))
   {
      (void) strcpy(fieldName, GetChar("Field name (): ", ""));
      fieldNamePtr = fieldName;
   }
   else
      fieldNamePtr = NULL;
   if (GetARBoolean("Have new option? (F): ", FALSE))
   {
      option = GetInt("Required, Optional, System, or Display-only (1-4) (2): ",
                      2);
      optionPtr = &option;
   }
   else
      optionPtr = NULL;
   if (GetARBoolean("Have new create mode? (F): ", FALSE))
   {
      createMode = GetInt("Open or Protected at create (1 or 2) (1): ", 1);
      createModePtr = &createMode;
   }
   else
      createModePtr = NULL;
   if ((dataType != AR_DATA_TYPE_ATTACH) &&
       (dataType != AR_DATA_TYPE_TABLE) &&
       (dataType != AR_DATA_TYPE_COLUMN) &&
       (GetARBoolean("Have new default value? (F): ", FALSE)))
   {
      defaultVal.dataType = dataType;
      GetValue(&defaultVal);
      defaultValPtr = &defaultVal;
   }
   else
      defaultValPtr = NULL;
   if (GetARBoolean("Have new permissions? (F): ", FALSE))
   {
      GetARPermissionList(TRUE, &permissions);
      permissionsPtr = &permissions;
   }
   else
      permissionsPtr = NULL;
   if (GetARBoolean("Have new field limits? (F): ", FALSE))
   {
      GetARFieldLimitStruct(dataType, &limit);
      limitPtr = &limit;
   }
   else
      limitPtr = NULL;
   if (GetARBoolean("Have new display instance info? (F): ", FALSE))
   {
      GetARDisplayInstanceList(&dInstanceList);
      dInstanceListPtr = &dInstanceList;
   }
   else
      dInstanceListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      (void) strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      (void) strcpy(owner, GetChar("Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      (void) strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   result = ARSetField(GetControlStructPtr(), schemaName, fieldId, fieldNamePtr,
                       optionPtr, createModePtr, defaultValPtr, permissionsPtr,
                       limitPtr, dInstanceListPtr, helpTextPtr, ownerPtr,
                       changeDiaryPtr, &status);

   /* Print results */
   DriverPrint("\n   ARSetField  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (defaultValPtr != NULL)
      FreeARValueStruct(&defaultVal, FALSE);
   if (permissionsPtr != NULL)
      FreeARPermissionList(&permissions, FALSE);
   if (limitPtr != NULL)
      FreeARFieldLimitStruct(&limit, FALSE);
   if (dInstanceListPtr != NULL)
      FreeARDisplayInstanceList(&dInstanceList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateField                             */
/*                                                                           */
/*****************************************************************************/

void APIARCreateField()

{
   char               changeDiary[1024];
   unsigned int       createMode;
   unsigned int       dataType;
   ARValueStruct      defaultVal;
   ARDisplayInstanceList dInstanceList;
   ARInternalId       fieldId;
   ARNameType         fieldName;
   char               helpText[1024];
   ARFieldLimitStruct limit;
   unsigned int       option;
   ARNameType         owner;
   ARPermissionList   permissions;
   int                result;
   ARBoolean          requiredIdOK;
   ARNameType         schemaName;
   ARStatusList       status;
   ARFieldMappingStruct fieldMap;

   /* Get Parameters */
   (void) printf("CREATE FIELD\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   fieldId = GetLong("Field id (0): ", 0);
   requiredIdOK = GetARBoolean("Create even if ID is reserved? (F): ", FALSE);
   (void) strcpy(fieldName, GetChar("Field name: ", ""));
   dataType = GetInt(
"Datatype Null/Key/Int/Real/Char/Diary/Enum/Time/Bitmask/Byte/Decimal/\n         attach/join/trim/control/table/column/ulong/coords\n         (0 - 11, 30-34, 40-41) (2): ",
                     AR_DATA_TYPE_INTEGER);
   option = GetInt("Required, Optional, System, or Display-only (1-4) (2): ",2);
   createMode = GetInt("Open or Protected at create (1 or 2) (1): ", 1);
   if ((dataType != AR_DATA_TYPE_ATTACH) &&
       (dataType != AR_DATA_TYPE_TABLE) &&
       (dataType != AR_DATA_TYPE_COLUMN) &&
       (GetARBoolean("Have default value? (T): ", TRUE)))
   {
      defaultVal.dataType = dataType;
      GetValue(&defaultVal);
   }
   else
   {
      memset((char *) &defaultVal, '\0', sizeof(defaultVal));
      defaultVal.dataType = AR_DEFAULT_VALUE_NONE;
   }
   GetARPermissionList(TRUE, &permissions);
   if (GetARBoolean("Have field limits? (T): ", TRUE))
      GetARFieldLimitStruct(dataType, &limit);
   else
      limit.dataType = AR_FIELD_LIMIT_NONE;
   GetARDisplayInstanceList(&dInstanceList);
   (void) strcpy(helpText, GetChar("Help Text: ", ""));
   (void) strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   (void) strcpy(changeDiary, GetChar("Change Diary: ", ""));
   GetARFieldMapping(&fieldMap);

   /* Call routine */
   result = ARCreateField(GetControlStructPtr(), schemaName, &fieldId,
                          requiredIdOK, fieldName, &fieldMap, dataType, option,
                          createMode, &defaultVal, &permissions, &limit,
                          &dInstanceList, helpText, owner, changeDiary,
                          &status);

   /* Print results */
   DriverPrint("\n   ARCreateField  results\n");
   PrintReturnCode(result);
   PrintARInternalId("Field id: ", fieldId);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARValueStruct(&defaultVal, FALSE);
   FreeARPermissionList(&permissions, FALSE);
   FreeARFieldLimitStruct(&limit, FALSE);
   FreeARDisplayInstanceList(&dInstanceList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteField                             */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteField()

{
   unsigned int     deleteOption;
   ARInternalId     fieldId;
   int              result;
   ARNameType       schemaName;
   ARStatusList     status;

   /* Get Parameters */
   (void) printf("DELETE FIELD\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   fieldId = GetLong("Field id (0): ", 0);
   deleteOption = GetInt("Delete option - clean, data, cascade (0 - 2) (0): ",
                         0);

   /* Call routine */
   result = ARDeleteField(GetControlStructPtr(), schemaName, fieldId,
                          deleteOption, &status);

   /* Print results */
   DriverPrint("\n   ARDeleteField  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                          APIARDeleteMultipleFields                        */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteMultipleFields()

{
   unsigned int     deleteOption;
   ARInternalIdList fieldIdList;
   int              result;
   ARNameType       schemaName;
   ARStatusList     status;

   /* Get Parameters */
   (void) printf("DELETE MULTIPLE FIELDS\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   (void) printf("Ids of fields to delete:\n");
   GetARInternalIdList(&fieldIdList);
   deleteOption = GetInt("Delete option - clean, data, cascade (0 - 2) (0): ",
                         0);
 
   /* Call routine */
   result = ARDeleteMultipleFields(GetControlStructPtr(), schemaName,
                                   &fieldIdList, deleteOption, &status);

   /* Print results */
   DriverPrint("\n   ARDeleteMultipleFields  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);
 
   /* Cleanup */
   FreeARInternalIdList(&fieldIdList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListField                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetListField()

{
   ARTimestamp        changedSince;
   ARInternalIdList   fieldIdList;
   int                result;
   ARNameType         schemaName;
   unsigned long      fieldType;
   ARStatusList       status;

   /* Get Parameters */
   (void) printf("GETLIST FIELD\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   changedSince = GetARTimestamp("Get all changed since (0): ", (ARTimestamp)0);
   fieldType = GetLong("Field Type - data, trim, control, table, column (1, 2, 4, 32, 64) (1): ", 1);

   /* Call routine */
   result = ARGetListField(GetControlStructPtr(), schemaName, fieldType,
                           changedSince, &fieldIdList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListField  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARInternalIdList("Field id list: ", "   ", &fieldIdList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARInternalIdList(&fieldIdList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetVUI                                  */
/*                                                                           */
/*****************************************************************************/

void APIARGetVUI()

{
   char              *changeDiary;
   ARPropList         dPropList;
   char              *helpText;
   ARNameType         lastChanged;
   ARNameType         owner;
   int                result;
   ARNameType         schemaName;
   ARStatusList       status;
   ARTimestamp        timestamp;
   ARInternalId       vuiId;
   ARNameType         vuiName;

   /* Get Parameters */
   (void) printf("GET VUI\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   vuiId = GetLong("VUI id (1): ", 1);

   /* Call routine */
   result = ARGetVUI(GetControlStructPtr(), schemaName, vuiId, vuiName,
                     &dPropList, &helpText, &timestamp, owner, lastChanged,
                     &changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARGetVUI  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintChar("VUI name: ", vuiName);
      PrintARPropList("", "", &dPropList);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARPropList(&dPropList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARSetVUI                                */
/*                                                                           */
/*****************************************************************************/

void APIARSetVUI()

{
   char                changeDiary[1024];
   char               *changeDiaryPtr;
   ARPropList          dPropList;
   ARPropList         *dPropListPtr;
   char                helpText[1024];
   char               *helpTextPtr;
   ARNameType          owner;
   char               *ownerPtr;
   int                 result;
   ARNameType          schemaName;
   ARStatusList        status;
   ARInternalId        vuiId;
   ARNameType          vuiName;
   char               *vuiNamePtr;

   /* Get Parameters */
   (void) printf("SET VUI\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   vuiId = GetLong("VUI id (1): ", 1);
   if (GetARBoolean("Change VUI name? (F): ", FALSE))
   {
      (void) strcpy(vuiName, GetChar("VUI name(): ", ""));
      vuiNamePtr = vuiName;
   }
   else
      vuiNamePtr = NULL;
   if (GetARBoolean("Have new Prop List info? (F): ", FALSE))
   {
      GetARPropList(&dPropList);
      dPropListPtr = &dPropList;
   }
   else
      dPropListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      (void) strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      (void) strcpy(owner, GetChar("Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      (void) strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   result = ARSetVUI(GetControlStructPtr(), schemaName, vuiId, vuiNamePtr,
                     dPropListPtr, helpTextPtr, ownerPtr, changeDiaryPtr,
                     &status);

   /* Print results */
   DriverPrint("\n   ARSetVUI  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (dPropListPtr != NULL)
      FreeARPropList(&dPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateVUI                               */
/*                                                                           */
/*****************************************************************************/

void APIARCreateVUI()

{
   char               changeDiary[1024];
   ARPropList         dPropList;
   char               helpText[1024];
   ARNameType         owner;
   int                result;
   ARNameType         schemaName;
   ARStatusList       status;
   ARInternalId       vuiId;
   ARNameType         vuiName;

   /* Get Parameters */
   (void) printf("CREATE VUI\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   vuiId = GetLong("VUI id (0): ", 0);
   (void) strcpy(vuiName, GetChar("VUI name: ", ""));
   GetARPropList(&dPropList);
   (void) strcpy(helpText, GetChar("Help Text: ", ""));
   (void) strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   (void) strcpy(changeDiary, GetChar("Change Diary: ", ""));

   /* Call routine */
   result = ARCreateVUI(GetControlStructPtr(), schemaName, &vuiId, vuiName,
                        &dPropList, helpText, owner, changeDiary, &status);
   /* Print results */
   DriverPrint("\n   ARCreateVUI  results\n");
   PrintReturnCode(result);
   PrintARInternalId("VUI id: ", vuiId);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARPropList(&dPropList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteVUI                               */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteVUI()

{
   int              result;
   ARNameType       schemaName;
   ARStatusList     status;
   ARInternalId     vuiId;

   /* Get Parameters */
   (void) printf("DELETE VUI\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   vuiId = GetLong("VUI id (0): ", 0);

   /* Call routine */
   result = ARDeleteVUI(GetControlStructPtr(), schemaName, vuiId, &status);

   /* Print results */
   DriverPrint("\n   ARDeleteVUI  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListVUI                              */
/*                                                                           */
/*****************************************************************************/

void APIARGetListVUI()

{
   ARTimestamp        changedSince;
   int                result;
   ARNameType         schemaName;
   ARStatusList       status;
   ARInternalIdList   vuiIdList;

   /* Get Parameters */
   (void) printf("GETLIST VUI\n");
   (void) strcpy(schemaName, GetChar("Schema name: ", ""));
   changedSince = GetARTimestamp("Get all changed since (0): ", (ARTimestamp)0);

   /* Call routine */
   result = ARGetListVUI(GetControlStructPtr(), schemaName, changedSince,
                         &vuiIdList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListVUI  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARInternalIdList("VUI id list: ", "   ", &vuiIdList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARInternalIdList(&vuiIdList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetCharMenu                             */
/*                                                                           */
/*****************************************************************************/

void APIARGetCharMenu()

{
   char              *changeDiary;
   char              *helpText;
   ARNameType         lastChanged;
   ARCharMenuStruct   menuDefn;
   ARNameType         name;
   ARNameType         owner;
   unsigned int       refreshCode;
   int                result;
   ARStatusList       status;
   ARTimestamp        timestamp;

   /* Get Parameters */
   (void) printf("GET CHAR MENU\n");
   (void) strcpy(name, GetChar("Character Menu name: ", ""));

   /* Call routine */
   result = ARGetCharMenu(GetControlStructPtr(), name, &refreshCode, &menuDefn,
                          &helpText, &timestamp, owner, lastChanged,
                          &changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARGetCharMenu  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintInt("Refresh code: ", refreshCode);
      PrintARCharMenuStruct("", &menuDefn);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARCharMenuStruct(&menuDefn, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARSetCharMenu                             */
/*                                                                           */
/*****************************************************************************/

void APIARSetCharMenu()

{
   char               changeDiary[1024];
   char              *changeDiaryPtr;
   char               helpText[1024];
   char              *helpTextPtr;
   ARCharMenuStruct   menuDefn;
   ARCharMenuStruct  *menuDefnPtr;
   ARNameType         name;
   ARNameType         newName;
   char              *newNamePtr;
   ARNameType         owner;
   char              *ownerPtr;
   unsigned int       refreshCode;
   unsigned int      *refreshCodePtr;
   int                result;
   ARStatusList       status;

   /* Get Parameters */
   (void) printf("SET CHAR MENU\n");
   (void) strcpy(name, GetChar("Character Menu name: ", ""));
   if (GetARBoolean("Change character menu name? (F): ", FALSE))
   {
      (void) strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change refresh code? (F): ", FALSE))
   {
      refreshCode = GetInt("Refresh on connect, open, interval (1 - 3) (1): ",
                           1);
      refreshCodePtr = &refreshCode;
   }
   else
      refreshCodePtr = NULL;
   if (GetARBoolean("Change menu definition(s)? (F): ", FALSE))
   {
      (void) printf("Character menu struct:\n");
      GetARCharMenuStruct(&menuDefn);
      menuDefnPtr = &menuDefn;
   }
   else
      menuDefnPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      (void) strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      (void) strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      (void) strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   result = ARSetCharMenu(GetControlStructPtr(), name, newNamePtr,
                          refreshCodePtr, menuDefnPtr, helpTextPtr, ownerPtr,
                          changeDiaryPtr, &status);

   /* Print results */
   DriverPrint("\n   ARSetCharMenu  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (menuDefnPtr != NULL)
      FreeARCharMenuStruct(&menuDefn, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARCreateCharMenu                            */
/*                                                                           */
/*****************************************************************************/

void APIARCreateCharMenu()

{
   char               changeDiary[1024];
   char               helpText[1024];
   ARCharMenuStruct   menuDefn;
   ARNameType         name;
   ARNameType         owner;
   unsigned int       refreshCode;
   int                result;
   ARStatusList       status;

   /* Get Parameters */
   (void) printf("CREATE CHAR MENU\n");
   (void) strcpy(name, GetChar("Character Menu name: ", ""));
   refreshCode = GetInt("Refresh on connect, open, interval (1 - 3) (1): ", 1);
   (void) printf("Character menu struct:\n");
   GetARCharMenuStruct(&menuDefn);
   (void) strcpy(helpText, GetChar("Help Text (): ", ""));
   (void) strcpy(owner, GetChar("Owner (): ", ""));
   (void) strcpy(changeDiary, GetChar("Change Diary (): ", ""));

   /* Call routine */
   result = ARCreateCharMenu(GetControlStructPtr(), name, refreshCode,
                             &menuDefn, helpText, owner, changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARCreateCharMenu  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARCharMenuStruct(&menuDefn, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARDeleteCharMenu                            */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteCharMenu()

{
   int                result;
   ARNameType         name;
   ARStatusList       status;

   /* Get Parameters */
   (void) printf("DELETE CHAR MENU\n");
   (void) strcpy(name, GetChar("Character Menu name: ", ""));

   /* Call routine */
   result = ARDeleteCharMenu(GetControlStructPtr(), name, &status);

   /* Print results */
   DriverPrint("\n   ARDeleteCharMenu  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARGetListCharMenu                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetListCharMenu()

{
   ARTimestamp          changedSince;
   ARNameList           nameList;
   int                  result;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("GETLIST CHAR MENU\n");
   changedSince = GetARTimestamp("Get all changed since (0): ", (ARTimestamp)0);

   /* Call routine */
   result = ARGetListCharMenu(GetControlStructPtr(), changedSince, &nameList,
                              &status);

   /* Print results */
   DriverPrint("\n   ARGetListCharMenu  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Character Menu list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                            APIARExpandCharMenu                            */
/*                                                                           */
/*****************************************************************************/

void APIARExpandCharMenu()

{
   ARCharMenuStruct     menuDefn;
   ARCharMenuStruct     menu;
   int                  result;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("EXPAND CHAR MENU\n");
   (void) printf("Character menu struct:\n");
   GetARCharMenuStruct(&menuDefn);

   /* Call routine */
   result = ARExpandCharMenu(GetControlStructPtr(), &menuDefn, &menu, &status);

   /* Print results */
   DriverPrint("\n   ARExpandCharMenu  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARCharMenuStruct("", &menu);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARCharMenuStruct(&menuDefn, FALSE);
   if (result < AR_RETURN_ERROR)
      FreeARCharMenuStruct(&menu, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                           FILTER                        **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                                 APIARGetFilter                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetFilter()

{
   ARFilterActionList   actionList;
   char                *changeDiary;
   ARFilterActionList   elseList;
   unsigned int         enable;
   char                *helpText;
   ARNameType           lastChanged;
   ARNameType           name;
   unsigned int         opSet;
   unsigned int         order;
   ARQualifierStruct    query;
   ARNameType           owner;
   int                  result;
   ARNameType           schema;
   ARStatusList         status;
   ARTimestamp          timestamp;

   /* Get Parameters */
   (void) printf("GET FILTER\n");
   (void) strcpy(name, GetChar("Filter name: ", ""));

   /* Call routine */
   result = ARGetFilter(GetControlStructPtr(), name, &order, schema, &opSet,
                        &enable, &query, &actionList, &elseList, &helpText,
                        &timestamp, owner, lastChanged, &changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARGetFilter  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintInt("Filter order: ", order);
      PrintChar("Schema name: ", schema);
      PrintInt("Operations bitmask: ", opSet);
      PrintInt("Enable flag : ", enable);
      PrintARQualifierStruct("", &query);
      PrintARFilterActionList(&actionList, TRUE);
      PrintARFilterActionList(&elseList, FALSE);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARQualifierStruct(&query, FALSE);
      FreeARFilterActionList(&actionList, FALSE);
      FreeARFilterActionList(&elseList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                 APIARSetFilter                            */
/*                                                                           */
/*****************************************************************************/

void APIARSetFilter()

{
   ARFilterActionList    actionList;
   ARFilterActionList   *actionListPtr;
   char                  changeDiary[1024];
   char                 *changeDiaryPtr;
   ARFilterActionList    elseList;
   ARFilterActionList   *elseListPtr;
   unsigned int          enable;
   unsigned int         *enablePtr;
   char                  helpText[1024];
   char                 *helpTextPtr;
   ARNameType            name;
   ARNameType            newName;
   char                 *newNamePtr;
   unsigned int          opSet;
   unsigned int         *opSetPtr;
   unsigned int          order;
   unsigned int         *orderPtr;
   ARNameType            owner;
   char                 *ownerPtr;
   ARQualifierStruct     query;
   ARQualifierStruct    *queryPtr;
   int                   result;
   ARNameType            schema;
   char                 *schemaPtr;
   ARStatusList          status;

   /* Get Parameters */
   (void) printf("SET FILTER\n");
   (void) strcpy(name, GetChar("Filter name: ", ""));
   if (GetARBoolean("Change filter name? (F): ", FALSE))
   {
      (void) strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change filter order? (F): ", FALSE))
   {
      order = GetInt("Filter order (0 - 1000) (500): ", 500);
      orderPtr = &order;
   }
   else
      orderPtr = NULL;
   if (GetARBoolean("Change schema? (F): ", FALSE))
   {
      (void) strcpy(schema, GetChar("Schema name (): ", ""));
      schemaPtr = schema;
   }
   else
      schemaPtr = NULL;
   if (GetARBoolean("Change operations mask? (F): ", FALSE))
   {
      opSet = GetInt("Operations bit mask (0): ", 0);
      opSetPtr = &opSet;
   }
   else
      opSetPtr = NULL;
   if (GetARBoolean("Change enable/disable? (F): ", FALSE))
   {
      enable = GetInt("Disable/enable (0-1) (1): ", 1);
      enablePtr = &enable;
   }
   else
      enablePtr = NULL;
   if (GetARBoolean("Change query? (F): ", FALSE))
   {
      GetARQualifierStruct(&query);
      queryPtr = &query;
   }
   else
      queryPtr = NULL;
   if (GetARBoolean("Change action(s)? (F): ", FALSE))
   {
      GetARFilterActionList(TRUE, &actionList);
      actionListPtr = &actionList;
   }
   else
      actionListPtr = NULL;
   if (GetARBoolean("Change else(s)? (F): ", FALSE))
   {
      GetARFilterActionList(FALSE, &elseList);
      elseListPtr = &elseList;
   }
   else
      elseListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      (void) strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      (void) strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      (void) strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   result = ARSetFilter(GetControlStructPtr(), name, newNamePtr, orderPtr,
                        schemaPtr, opSetPtr, enablePtr, queryPtr, actionListPtr,
                        elseListPtr, helpTextPtr, ownerPtr, changeDiaryPtr,
                        &status);

   /* Print results */
   DriverPrint("\n   ARSetFilter  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (queryPtr != NULL)
      FreeARQualifierStruct(&query, FALSE);
   if (actionListPtr != NULL)
      FreeARFilterActionList(&actionList, FALSE);
   if (elseListPtr != NULL)
      FreeARFilterActionList(&elseList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                               APIARCreateFilter                           */
/*                                                                           */
/*****************************************************************************/

void APIARCreateFilter()

{
   ARFilterActionList   actionList;
   char                 changeDiary[1024];
   ARFilterActionList   elseList;
   unsigned int         enable;
   char                 helpText[1024];
   ARNameType           name;
   unsigned int         opSet;
   unsigned int         order;
   ARNameType           owner;
   ARQualifierStruct    query;
   int                  result;
   ARNameType           schema;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("CREATE FILTER\n");
   (void) strcpy(name, GetChar("Filter name: ", ""));
   order = GetInt("Filter order (0 - 1000) (500): ", 500);
   (void) strcpy(schema, GetChar("Schema name: ", ""));
   opSet = GetInt("Operations bit mask (0): ", 0);
   enable = GetInt("Disable/Enable (0-1) (1): ", 1);
   GetARQualifierStruct(&query);
   GetARFilterActionList(TRUE, &actionList);
   GetARFilterActionList(FALSE, &elseList);
   (void) strcpy(helpText, GetChar("Help Text: ", ""));
   (void) strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   (void) strcpy(changeDiary, GetChar("Change Diary: ", ""));

   /* Call routine */
   result = ARCreateFilter(GetControlStructPtr(), name, order, schema, opSet,
                           enable, &query, &actionList, &elseList, helpText,
                           owner, changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARCreateFilter  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&query, FALSE);
   FreeARFilterActionList(&actionList, FALSE);
   FreeARFilterActionList(&elseList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                               APIARDeleteFilter                           */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteFilter()

{
   ARNameType           name;
   int                  result;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("DELETE FILTER\n");
   (void) strcpy(name, GetChar("Filter name: ", ""));

   /* Call routine */
   result = ARDeleteFilter(GetControlStructPtr(), name, &status);

   /* Print results */
   DriverPrint("\n   ARDeleteFilter  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                               APIARGetListFilter                          */
/*                                                                           */
/*****************************************************************************/

void APIARGetListFilter()

{
   ARTimestamp          changedSince;
   ARNameList           nameList;
   int                  result;
   ARNameType           schema;
   char                *schemaPtr;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("GETLIST FILTER\n");
   if (GetARBoolean("For a specific schema? (F): ", FALSE))
   {
      (void) strcpy(schema, GetChar("Schema name (): ", ""));
      schemaPtr = schema;
   }
   else
      schemaPtr = NULL;
   changedSince = GetARTimestamp("Get all changed since (0): ", (ARTimestamp)0);

   /* Call routine */
   result = ARGetListFilter(GetControlStructPtr(), schemaPtr, changedSince,
                            &nameList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListFilter  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Filter list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                        ACTIVE LINK                      **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                                APIARGetActiveLink                         */
/*                                                                           */
/*****************************************************************************/

void APIARGetActiveLink()

{
   ARActiveLinkActionList actionList;
   char                  *changeDiary;
   ARActiveLinkActionList elseList;
   unsigned int           enable;
   unsigned int           executeMask;
   ARInternalId           controlfield;
   ARInternalId           focusfield;
   ARInternalIdList       groupList;
   char                  *helpText;
   ARNameType             lastChanged;
   ARNameType             name;
   unsigned int           order;
   ARNameType             owner;
   ARQualifierStruct      query;
   int                    result;
   ARNameType             schema;
   ARStatusList           status;
   ARTimestamp            timestamp;

   /* Get Parameters */
   (void) printf("GET ACTIVE LINK\n");
   (void) strcpy(name, GetChar("Active link name: ", ""));

   /* Call routine */
   result = ARGetActiveLink(GetControlStructPtr(), name, &order, schema,
                            &groupList, &executeMask, &controlfield,
                            &focusfield, &enable, &query, &actionList,
                            &elseList, &helpText, &timestamp, owner,
                            lastChanged, &changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARGetActiveLink  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintInt("Execution order: ", order);
      PrintChar("Schema: ", schema);
      PrintARInternalIdList("Group list: ", "   ", &groupList);
      PrintInt("Execute bitmask: ", executeMask);
      PrintARInternalId("Control Field: ", controlfield);
      PrintARInternalId("Focus Field: ", focusfield);
      PrintInt("Enable flag : ", enable);
      PrintARQualifierStruct("", &query);
      PrintARActiveLinkActionList(&actionList, TRUE);
      PrintARActiveLinkActionList(&elseList, FALSE);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARInternalIdList(&groupList, FALSE);
      FreeARQualifierStruct(&query, FALSE);
      FreeARActiveLinkActionList(&actionList, FALSE);
      FreeARActiveLinkActionList(&elseList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARSetActiveLink                         */
/*                                                                           */
/*****************************************************************************/

void APIARSetActiveLink()

{
   ARActiveLinkActionList     actionList;
   ARActiveLinkActionList    *actionListPtr;
   char                       changeDiary[1024];
   char                      *changeDiaryPtr;
   ARActiveLinkActionList     elseList;
   ARActiveLinkActionList    *elseListPtr;
   unsigned int               enable;
   unsigned int              *enablePtr;
   unsigned int               executeMask;
   unsigned int              *executeMaskPtr;
   ARInternalId               controlfield;
   ARInternalId               focusfield;
   ARInternalId              *controlfieldPtr;
   ARInternalId              *focusfieldPtr;
   ARInternalIdList           groupList;
   ARInternalIdList          *groupListPtr;
   char                       helpText[1024];
   char                      *helpTextPtr;
   ARNameType                 name;
   ARNameType                 newName;
   char                      *newNamePtr;
   unsigned int               order;
   unsigned int              *orderPtr;
   ARNameType                 owner;
   char                      *ownerPtr;
   ARQualifierStruct          query;
   ARQualifierStruct         *queryPtr;
   int                        result;
   ARNameType                 schema;
   char                      *schemaPtr;
   ARStatusList               status;

   /* Get Parameters */
   (void) printf("SET ACTIVE LINK\n");
   (void) strcpy(name, GetChar("Active link name: ", ""));
   if (GetARBoolean("Change active link name? (F): ", FALSE))
   {
      (void) strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change execution order? (F): ", FALSE))
   {
      order = GetInt("Execution order (1 - 1000) (1): ", 1);
      orderPtr = &order;
   }
   else
      orderPtr = NULL;
   if (GetARBoolean("Change schema name? (F): ", FALSE))
   {
      (void) strcpy(schema, GetChar("Schema name (): ", ""));
      schemaPtr = schema;
   }
   else
      schemaPtr = NULL;
   if (GetARBoolean("Change group list? (F): ", FALSE))
   {
      (void) printf("Ids of groups allowed to perform active link:\n");
      GetARInternalIdList(&groupList);
      groupListPtr = &groupList;
   }
   else
      groupListPtr = NULL;
   if (GetARBoolean("Change execute mask? (F): ", FALSE))
   {
      executeMask = GetInt("Execute bit mask (1): ", 1);
      executeMaskPtr = &executeMask;
   }
   else
      executeMaskPtr = NULL;
   if (GetARBoolean("Change control field? (F): ", FALSE))
   {
      controlfield = GetLong("Control Field to tie active link to (0): ", 0);
      controlfieldPtr = &controlfield;
   }
   else
      controlfieldPtr = NULL;
   if (GetARBoolean("Change focus field? (F): ", FALSE))
   {
      focusfield = GetLong("Focus Field to tie active link to (0): ", 0);
      focusfieldPtr = &focusfield;
   }
   else
      focusfieldPtr = NULL;
   if (GetARBoolean("Change enable/disable? (F): ", FALSE))
   {
      enable = GetInt("Disable/enable (0-1) (1): ", 1);
      enablePtr = &enable;
   }
   else
      enablePtr = NULL;
   if (GetARBoolean("Change query? (F): ", FALSE))
   {
      GetARQualifierStruct(&query);
      queryPtr = &query;
   }
   else
      queryPtr = NULL;
   if (GetARBoolean("Change action(s)? (F): ", FALSE))
   {
      GetARActiveLinkActionList(TRUE, &actionList);
      actionListPtr = &actionList;
   }
   else
      actionListPtr = NULL;
   if (GetARBoolean("Change else(s)? (F): ", FALSE))
   {
      GetARActiveLinkActionList(FALSE, &elseList);
      elseListPtr = &elseList;
   }
   else
      elseListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      (void) strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      (void) strcpy(owner, GetChar("Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      (void) strcpy(changeDiary, GetChar("Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   result = ARSetActiveLink(GetControlStructPtr(), name, newNamePtr, orderPtr,
                            schemaPtr, groupListPtr, executeMaskPtr,
                            controlfieldPtr, focusfieldPtr, enablePtr, queryPtr,
                            actionListPtr, elseListPtr, helpTextPtr, ownerPtr,
                            changeDiaryPtr, &status);

   /* Print results */
   DriverPrint("\n   ARSetActiveLink  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (groupListPtr != NULL)
      FreeARInternalIdList(&groupList, FALSE);
   if (queryPtr != NULL)
      FreeARQualifierStruct(&query, FALSE);
   if (actionListPtr != NULL)
      FreeARActiveLinkActionList(&actionList, FALSE);
   if (elseListPtr != NULL)
      FreeARActiveLinkActionList(&elseList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateActiveLink                        */
/*                                                                           */
/*****************************************************************************/

void APIARCreateActiveLink()

{
   ARActiveLinkActionList    actionList;
   char                      changeDiary[1024];
   ARActiveLinkActionList    elseList;
   unsigned int              enable;
   unsigned int              executeMask;
   ARInternalId              controlfield;
   ARInternalId              focusfield;
   ARInternalIdList          groupList;
   char                      helpText[1024];
   ARNameType                name;
   unsigned int              order;
   ARNameType                owner;
   ARQualifierStruct         query;
   int                       result;
   ARNameType                schema;
   ARStatusList              status;

   /* Get Parameters */
   (void) printf("CREATE ACTIVE LINK\n");
   (void) strcpy(name, GetChar("Active link name: ", ""));
   order = GetInt("Execution order (1 - 1000) (1): ", 1);
   (void) strcpy(schema, GetChar("Schema name: ", ""));
   (void) printf("Ids of groups allowed to perform active link:\n");
   GetARInternalIdList(&groupList);
   executeMask = GetInt("Execute bit mask (1): ", 1);
   controlfield = GetLong("Control Field to tie active link to (0): ", 0);
   focusfield = GetLong("Focus Field to tie active link to (0): ", 0);
   enable = GetInt("Disable/Enable (0-1) (1): ", 1);
   GetARQualifierStruct(&query);
   GetARActiveLinkActionList(TRUE, &actionList);
   GetARActiveLinkActionList(FALSE, &elseList);
   (void) strcpy(helpText, GetChar("Help Text: ", ""));
   (void) strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   (void) strcpy(changeDiary, GetChar("Change Diary: ", ""));

   /* Call routine */
   result = ARCreateActiveLink(GetControlStructPtr(), name, order, schema,
                               &groupList, executeMask, &controlfield,
                               &focusfield, enable, &query, &actionList,
                               &elseList, helpText, owner, changeDiary,
                               &status);

   /* Print results */
   DriverPrint("\n   ARCreateActiveLink  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARInternalIdList(&groupList, FALSE);
   FreeARQualifierStruct(&query, FALSE);
   FreeARActiveLinkActionList(&actionList, FALSE);
   FreeARActiveLinkActionList(&elseList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteActiveLink                        */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteActiveLink()

{
   ARNameType           name;
   int                  result;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("DELETE ACTIVE LINK\n");
   (void) strcpy(name, GetChar("Active link name: ", ""));

   /* Call routine */
   result = ARDeleteActiveLink(GetControlStructPtr(), name, &status);

   /* Print results */
   DriverPrint("\n   ARDeleteActiveLink  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListActiveLink                       */
/*                                                                           */
/*****************************************************************************/

void APIARGetListActiveLink()

{
   ARTimestamp          changedSince;
   ARNameList           nameList;
   int                  result;
   ARNameType           schema;
   char                *schemaPtr;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("GETLIST ACTIVE LINK\n");
   if (GetARBoolean("For a specific schema? (F): ", FALSE))
   {
      (void) strcpy(schema, GetChar("Schema name (): ", ""));
      schemaPtr = schema;
   }
   else
      schemaPtr = NULL;
   changedSince = GetARTimestamp("Get all changed since (0): ", (ARTimestamp)0);

   /* Call routine */
   result = ARGetListActiveLink(GetControlStructPtr(), schemaPtr, changedSince,
                                &nameList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListActiveLink  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Active link list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                        SUPPORT FILE                     **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                                APIARGetSupportFile                        */
/*                                                                           */
/*****************************************************************************/

void APIARGetSupportFile()

{
   ARInternalId           fileId;
   char                   filename[256];
   FILE                  *filePtr;
   unsigned int           fileType;
   ARInternalId           id2;
   ARNameType             name;
   int                    result;
   ARStatusList           status;
   ARTimestamp            timestamp;

   /* Get Parameters */
   (void) printf("GET SUPPORT FILE\n");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   (void) strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   fileId = GetLong("ID for the file (1): ", 1);

   filePtr = NULL;
   while (filePtr == NULL)
   {
      (void) strcpy(filename, GetChar("Filename to hold returned file: ", ""));
#ifdef _WIN32
      filePtr = fopen(filename, "wb");
#else /* _WIN32 */
      filePtr = fopen(filename, "w");
#endif /* _WIN32 */
      if (filePtr == NULL)
         (void) printf("   ***  File open failure\n");
   }

   /* Call routine */
   result = ARGetSupportFile(GetControlStructPtr(), fileType, name, id2, fileId,
                             filePtr, &timestamp, &status);

   /* Print results */
   DriverPrint("\n   ARGetSupportFile  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARTimestamp("Last update time: ", timestamp);
   PrintARStatusList(&status);

   /* Cleanup */
   (void) fclose(filePtr);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARSetSupportFile                        */
/*                                                                           */
/*****************************************************************************/

void APIARSetSupportFile()

{
   ARInternalId           fileId;
   char                   filename[256];
   FILE                  *filePtr;
   unsigned int           fileType;
   ARInternalId           id2;
   ARNameType             name;
   int                    result;
   ARStatusList           status;

   /* Get Parameters */
   (void) printf("SET SUPPORT FILE\n");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   (void) strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   fileId = GetLong("ID for the file (1): ", 1);

   filePtr = NULL;
   while (filePtr == NULL)
   {
      (void) strcpy(filename, GetChar("Filename to load to server: ", ""));
#ifdef _WIN32
      filePtr = fopen(filename, "rb");
#else /* _WIN32 */
      filePtr = fopen(filename, "r");
#endif /* _WIN32 */
      if (filePtr == NULL)
         (void) printf("   ***  File open failure\n");
   }

   /* Call routine */
   result = ARSetSupportFile(GetControlStructPtr(), fileType, name, id2, fileId,
                             filePtr, &status);

   /* Print results */
   DriverPrint("\n   ARSetSupportFile  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   (void) fclose(filePtr);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARCreateSupportFile                       */
/*                                                                           */
/*****************************************************************************/

void APIARCreateSupportFile()

{
   ARInternalId           fileId;
   char                   filename[256];
   FILE                  *filePtr;
   unsigned int           fileType;
   ARInternalId           id2;
   ARNameType             name;
   int                    result;
   ARStatusList           status;

   /* Get Parameters */
   (void) printf("CREATE SUPPORT FILE\n");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   (void) strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   fileId = GetLong("ID for the file (1): ", 1);

   filePtr = NULL;
   while (filePtr == NULL)
   {
      (void) strcpy(filename, GetChar("Filename to hold returned file: ", ""));
#ifdef _WIN32
      filePtr = fopen(filename, "rb");
#else /* _WIN32 */
      filePtr = fopen(filename, "r");
#endif /* _WIN32 */
      if (filePtr == NULL)
         (void) printf("   ***  File open failure\n");
   }

   /* Call routine */
   result = ARCreateSupportFile(GetControlStructPtr(), fileType, name, id2,
                                fileId, filePtr, &status);

   /* Print results */
   DriverPrint("\n   ARCreateSupportFile  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   (void) fclose(filePtr);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARDeleteSupportFile                       */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteSupportFile()

{
   ARInternalId           fileId;
   unsigned int           fileType;
   ARInternalId           id2;
   ARNameType             name;
   int                    result;
   ARStatusList           status;

   /* Get Parameters */
   (void) printf("DELETE SUPPORT FILE\n");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   (void) strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   fileId = GetLong("ID for the file (1): ", 1);

   /* Call routine */
   result = ARDeleteSupportFile(GetControlStructPtr(), fileType, name, id2,
                                fileId, &status);

   /* Print results */
   DriverPrint("\n   ARDeleteSupportFile  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListSupportFile                      */
/*                                                                           */
/*****************************************************************************/

void APIARGetListSupportFile()

{
   ARTimestamp            changedSince;
   ARInternalIdList       fileIdList;
   unsigned int           fileType;
   ARInternalId           id2;
   ARNameType             name;
   int                    result;
   ARStatusList           status;

   /* Get Parameters */
   (void) printf("GETLIST SUPPORT FILE\n");
   fileType = GetInt("File type -- external report (1) (1): ", 1);
   (void) strcpy(name, GetChar("Name of associated object: ", ""));
   id2 = GetLong("Supporting ID for object (0): ", 0);
   changedSince = GetARTimestamp("Get all changed since (0): ", (ARTimestamp)0);

   /* Call routine */
   result = ARGetListSupportFile(GetControlStructPtr(), fileType, name, id2,
                                 changedSince, &fileIdList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListSupportFile  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARInternalIdList("File ID list: ", "   ", &fileIdList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARInternalIdList(&fileIdList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                         GROUP/USER                      **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                               APIARGetListGroup                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetListGroup()

{
   ARGroupInfoList      groupList;
   int                  result;
   ARStatusList         status;
   ARNameType           user;
   char                *userPtr;

   /* Get Parameters */
   (void) printf("GETLIST GROUP\n");

   /* Call routine */
   if (GetARBoolean("For a specific user? (F): ", FALSE))
   {
      (void) strcpy(user, GetChar("Name of the user: ",
                                  GetControlStructPtr()->user));
      userPtr = user;
   }
   else
      userPtr = NULL;
   result = ARGetListGroup(GetControlStructPtr(), userPtr, &groupList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListGroup  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARGroupInfoList(&groupList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARGroupInfoList(&groupList, FALSE);
   FreeARStatusList(&status, FALSE);
}

/*****************************************************************************/
/*                                                                           */
/*                               APIARGetListUser                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetListUser()

{
   ARUserInfoList       userList;
   int                  result;
   ARStatusList         status;
   unsigned int         userListType;

   /* Get Parameters */
   (void) printf("GETLIST USER\n");
   userListType = GetInt(
                 "User List Type (myself, registered or current (0-2)(1): ", 1);

   /* Call routine */
   result = ARGetListUser(GetControlStructPtr(), userListType, &userList,
                          &status);

   /* Print results */
   DriverPrint("\n   ARGetListUser  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARUserInfoList(&userList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARUserInfoList(&userList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                     MISCELLANEOUS                       **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                               APIARVerifyUser                             */
/*                                                                           */
/*****************************************************************************/

void APIARVerifyUser()

{
   ARBoolean        adminFlag;
   ARBoolean        subadminFlag;
   ARBoolean        customFlag;
   int              result;
   ARStatusList     status;

   /* Get Parameters */
   (void) printf("VERIFY USER\n");

   /* Call routine */
   result = ARVerifyUser(GetControlStructPtr(), &adminFlag, &subadminFlag,
                         &customFlag, &status);

   /* Print results */
   DriverPrint("\n   ARVerifyUser  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARBoolean("Admin flag: ", adminFlag);
      PrintARBoolean("SubAdmin flag: ", subadminFlag);
      PrintARBoolean("Customize flag: ", customFlag);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                  APIARExport                              */
/*                                                                           */
/*****************************************************************************/

void APIARExport()

{
   char            *buff;
   ARNameType       displayTag;
   char             filename[256];
   FILE            *filePtr;
   int              result;
   ARStatusList     status;
   ARStructItemList structItems;

   status.numItems = 0;
   status.statusList = NULL;;

   /* Get Parameters */
   (void) printf("EXPORT\n");
   GetARStructItemList(&structItems);
   (void) strcpy(displayTag, GetChar("Display Tag: ", ""));
   filePtr = NULL;
   while (filePtr == NULL)
   {
      (void) strcpy(filename, GetChar("Filename for exported data: ", ""));
      filePtr = fopen(filename, "w");
      if (filePtr == NULL)
         (void) printf("   ***  File open failure\n");
   }

   /* Call routine */
   result = ARExport(GetControlStructPtr(), &structItems, displayTag, &buff,
                     &status);

   /* Print results */
   DriverPrint("\n   ARExport  results\n");
   PrintReturnCode(result);
   if ((result < AR_RETURN_ERROR) && (buff != NULL))
      (void) fwrite(buff, strlen(buff), 1, filePtr);
   (void) fclose(filePtr);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      if (buff != NULL)
         free(buff);
   else
      (void) unlink(filename);
   FreeARStructItemList(&structItems, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                  APIARImport                              */
/*                                                                           */
/*****************************************************************************/

void APIARImport()

{
   long             fileLen;
   char             filename[256];
   FILE            *filePtr;
   char            *importBuf;
   int              result;
   int              amtRead;
   int              amtRemaining;
   ARStatusList     status;
   ARStructItemList structItems;

   /* Get Parameters */
   (void) printf("IMPORT\n");
   GetARStructItemList(&structItems);
   filePtr = NULL;
   while (filePtr == NULL)
   {
      (void) strcpy(filename, GetChar("Filename containing import data: ", ""));
      filePtr = fopen(filename, "r");
      if (filePtr == NULL)
         (void) printf("   ***  File open failure\n");
   }
   (void) fseek(filePtr, 0L, 2);
   fileLen = ftell(filePtr);
   importBuf = (char *) malloc((unsigned) (fileLen + 1));
   if (importBuf == NULL)
   {
      (void) fclose(filePtr);
      DriverPrint("**** malloc Failure in import operation ****\n");
      return;
   }
   rewind(filePtr);

   /* Note that the length of the file (fileLen) may actually be */
   /* more than what can be read.  Some standard io libraries    */
   /* will compress carriage returns (\r) and new lines (\n) on  */
   /* fread(), but this may not be reflected in ftell().         */
   for (amtRead = 0, amtRemaining = fileLen; amtRead < fileLen; )
   {
      result = fread(&importBuf[amtRead], 1, amtRemaining, filePtr);
      if (result <= 0)
         break;
      amtRemaining -= result;
      amtRead += result;
   }
   importBuf[amtRead] = '\0';
   (void) fclose(filePtr);

   /* Call routine */
   result = ARImport(GetControlStructPtr(), &structItems, importBuf, &status);

   /* Print results */
   DriverPrint("\n   ARImport  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   (void) free(importBuf);
   FreeARStructItemList(&structItems, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListServer                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetListServer()

{
   ARServerNameList     nameList;
   int                  result;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("GETLIST SERVER\n");

   /* Call routine */
   result = ARGetListServer(GetControlStructPtr(), &nameList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListServer  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARServerNameList("Server list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARServerNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARInitialization                          */
/*                                                                           */
/*****************************************************************************/

void APIARInitialization()

{
   int                  result;
   ARStatusList         status;

   (void) printf("INITIALIZATION\n");

   /* Call routine */
   result = ARInitialization(GetControlStructPtr(), &status);

   /* Print results */
   DriverPrint("\n   ARInitialization  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                                APIARTermination                           */
/*                                                                           */
/*****************************************************************************/

void APIARTermination()

{
   int                  result;
   ARStatusList         status;

   (void) printf("TERMINATION\n");

   /* Call routine */
   result = ARTermination(GetControlStructPtr(), &status);

   /* Print results */
   DriverPrint("\n   ARTermination  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetServerInfo                           */
/*                                                                           */
/*****************************************************************************/

void APIARGetServerInfo()

{
   int                     result;
   ARServerInfoRequestList requestList;
   ARServerInfoList        serverInfoList;
   ARStatusList            status;

   /* Get Parameters */
   (void) printf("GETSERVERINFO\n");
   GetARServerInfoRequestList(&requestList);

   /* Call routine */
   result = ARGetServerInfo(GetControlStructPtr(), &requestList,
                            &serverInfoList, &status);

   /* Print results */
   DriverPrint("\n   ARGetServerInfo  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARServerInfoList(&serverInfoList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARServerInfoList(&serverInfoList, FALSE);
   FreeARServerInfoRequestList(&requestList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARSetServerInfo                           */
/*                                                                           */
/*****************************************************************************/

void APIARSetServerInfo()

{
   int                     result;
   ARServerInfoList        serverInfoList;
   ARStatusList            status;

   /* Get Parameters */
   (void) printf("SETSERVERINFO\n");
   GetARServerInfoList(&serverInfoList);

   /* Call routine */
   result = ARSetServerInfo(GetControlStructPtr(), &serverInfoList, &status);

   /* Print results */
   DriverPrint("\n   ARSetServerInfo  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARServerInfoList(&serverInfoList, FALSE);
   FreeARStatusList(&status, FALSE);
}

/*****************************************************************************/
/*                                                                           */
/*                              APIARSetServerPort                           */
/*                                                                           */
/*****************************************************************************/

void APIARSetServerPort()

{
   unsigned int  portNumber;
   unsigned int  privatSocket;
   int           result;
   ARStatusList  status;

   /* Get port number */
   (void) printf("SET SERVER PORT\n");
   portNumber = GetInt("The port number of server (0): ", 0);

   /* Get private socket number */
   privatSocket = GetInt("Using private socket? (0): ", 0);

   /* Call routine */
   result = ARSetServerPort(GetControlStructPtr(),
                            GetControlStructPtr()->server, portNumber,
                            privatSocket, &status);

   /* Print results */
   DriverPrint("\n   ARSetServerPort  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}



/*****************************************************************************/
/*                                                                           */
/*                            APIARGetServerStatistics                       */
/*                                                                           */
/*****************************************************************************/

void APIARGetServerStatistics()

{
   int                     result;
   ARServerInfoRequestList requestList;
   ARServerInfoList        serverInfoList;
   ARStatusList            status;

   /* Get Parameters */
   (void) printf("GETSERVERSTATISTICS\n");
   GetARServerInfoRequestList(&requestList);

   /* Call routine */
   result = ARGetServerStatistics(GetControlStructPtr(), &requestList,
                                  &serverInfoList, &status);

   /* Print results */
   DriverPrint("\n   ARGetServerStatistics  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARServerInfoList2(&serverInfoList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARServerInfoList(&serverInfoList, FALSE);
   FreeARServerInfoRequestList(&requestList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                          APIARGetTextForErrorMessage                      */
/*                                                                           */
/*****************************************************************************/

void APIARGetTextForErrorMessage()

{
   char  *msgBuf;
   int    msgId;

   /* Get message identifier */
   (void) printf("GET TEXT FOR ERROR MESSAGE\n");
   msgId = GetInt("The message identifier for the message (0): ", 0);

   /* Call routine */
   msgBuf = ARGetTextForErrorMessage(msgId);

   /* Print results */
   DriverPrint("\n   ARGetTextForErrorMessage  results\n");
   PrintChar("Message Text: ", msgBuf);
   
   /* Cleanup */
   if (msgBuf)
      free(msgBuf);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                  FULL TEXT INFORMATION                  **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                              APIARGetFullTextInfo                         */
/*                                                                           */
/*****************************************************************************/

void APIARGetFullTextInfo()

{
   int                       result;
   ARFullTextInfoRequestList requestList;
   ARFullTextInfoList        fullTextInfoList;
   ARStatusList              status;

   /* Get Parameters */
   (void) printf("GETFULLTEXTINFO\n");
   GetARFullTextInfoRequestList(&requestList);

   /* Call routine */
   result = ARGetFullTextInfo(GetControlStructPtr(), &requestList,
                              &fullTextInfoList, &status);

   /* Print results */
   DriverPrint("\n   ARGetFullTextInfo  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARFullTextInfoList(&fullTextInfoList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARFullTextInfoList(&fullTextInfoList, FALSE);
   FreeARStatusList(&status, FALSE);
}

/*****************************************************************************/
/*                                                                           */
/*                              APIARSetFullTextInfo                         */
/*                                                                           */
/*****************************************************************************/

void APIARSetFullTextInfo()

{
   int                     result;
   ARFullTextInfoList      fullTextInfoList;
   ARStatusList            status;

   /* Get Parameters */
   (void) printf("SETFULLTEXTINFO\n");
   GetARFullTextInfoList(&fullTextInfoList);

   /* Call routine */
   result = ARSetFullTextInfo(GetControlStructPtr(), &fullTextInfoList,
                              &status);

   /* Print results */
   DriverPrint("\n   ARSetFullTextInfo  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARFullTextInfoList(&fullTextInfoList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARGetListSQL                              */
/*                                                                           */
/*****************************************************************************/

void APIARGetListSQL()

{
   unsigned int      maxRetrieve;
   unsigned int      numMatches;
   unsigned int     *numMatchesPtr;
   int               result;
   char             *sqlCommand;
   ARStatusList      status;
   ARValueListList   valueListList;

   /* Get Parameters */
   (void) printf("GETLISTSQL\n");
   GetMallocChar("SQL command: ", "", &sqlCommand);
   maxRetrieve = GetInt("Maximum number of entries to retrieve (500): ", 500);
   if (GetARBoolean("Get number of matches? (F): ", FALSE))
      numMatchesPtr = &numMatches;
   else
      numMatchesPtr = NULL;

   /* Call routine */
   result = ARGetListSQL(GetControlStructPtr(), sqlCommand, maxRetrieve,
                         &valueListList, numMatchesPtr, &status);

   /* Print results */
   DriverPrint("\n   ARGetListSQL  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      PrintARValueListList(&valueListList);
      if (numMatchesPtr != NULL)
          PrintInt("Number of matches: ", numMatches);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (sqlCommand != NULL)
      (void) free(sqlCommand);
   if (result < AR_RETURN_ERROR)
      FreeARValueListList(&valueListList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                              APIARExecuteProcess                          */
/*                                                                           */
/*****************************************************************************/

void APIARExecuteProcess()

{
   char                 command[1024];
   int                  result;
   int                  returnStatus;
   int                 *returnStatusPtr;
   char                *returnString;
   char               **returnStringPtr;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("EXECUTE PROCESS\n");
   (void) strcpy(command, GetChar("Command: ", ""));
   if (GetARBoolean("Wait for process to complete? (F): ", FALSE))
   {
      returnStatusPtr = &returnStatus;
      returnStringPtr = &returnString;
   }
   else
   {
      returnStatusPtr = NULL;
      returnStringPtr = NULL;
   }

   /* Call routine */
   result = ARExecuteProcess(GetControlStructPtr(), command, returnStatusPtr,
                             returnStringPtr, &status);

   /* Print results */
   DriverPrint("\n   ARExecuteProcess  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      if (returnStatusPtr != NULL)
         PrintInt("Return Status: ", *returnStatusPtr);
      if (returnStringPtr != NULL)
         PrintChar("Process output: ", *returnStringPtr);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/**********                                                         **********/
/**********                       ESCALATION                        **********/
/**********                                                         **********/
/*****************************************************************************/

/*****************************************************************************/
/*                                                                           */
/*                             APIARGetEscalation                            */
/*                                                                           */
/*****************************************************************************/

void APIARGetEscalation()

{
   ARFilterActionList   actionList;
   char                *changeDiary;
   ARFilterActionList   elseList;
   char                *helpText;
   ARNameType           lastChanged;
   ARNameType           name;
   unsigned int         enable;
   AREscalationTmStruct escalationTm;
   ARQualifierStruct    query;
   ARNameType           owner;
   int                  result;
   ARNameType           schema;
   ARStatusList         status;
   ARTimestamp          timestamp;

   /* Get Parameters */
   (void) printf("GET ESCALATION\n");
   (void) strcpy(name, GetChar("Escalation name: ", ""));

   /* Call routine */
   result = ARGetEscalation(GetControlStructPtr(), name, &escalationTm, schema,
                            &enable, &query, &actionList, &elseList, &helpText,
                            &timestamp, owner, lastChanged, &changeDiary,
                            &status);

   /* Print results */
   DriverPrint("\n   ARGetEscalation  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
   {
      if (escalationTm.escalationTmType == AR_ESCALATION_TYPE_INTERVAL)
        PrintInt("Escalation interval: ", escalationTm.u.interval);
      else
      {
        PrintInt("Escalation monthday: ", escalationTm.u.date.monthday);
        PrintInt("Escalation weekday: ", escalationTm.u.date.weekday);
        PrintInt("Escalation hourmask: ", escalationTm.u.date.hourmask);
        PrintInt("Escalation minute: ", escalationTm.u.date.minute);
      }
      PrintChar("Schema name: ", schema);
      PrintInt("Enable flag : ", enable);
      PrintARQualifierStruct("", &query);
      PrintARFilterActionList(&actionList, TRUE);
      PrintARFilterActionList(&elseList, FALSE);
      PrintChar("Help text: ", helpText);
      PrintARTimestamp("Last update time: ", timestamp);
      PrintChar("Owner: ", owner);
      PrintChar("Last changed: ", lastChanged);
      APIPrintDiary("Change Diary: ", changeDiary);
   }
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
   {
      FreeARQualifierStruct(&query, FALSE);
      FreeARFilterActionList(&actionList, FALSE);
      FreeARFilterActionList(&elseList, FALSE);
      if (helpText != NULL)
         free(helpText);
      if (changeDiary != NULL)
         free(changeDiary);
   }
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                             APIARSetEscalation                            */
/*                                                                           */
/*****************************************************************************/

void APIARSetEscalation()

{
   ARFilterActionList    actionList;
   ARFilterActionList   *actionListPtr;
   char                  changeDiary[1024];
   char                 *changeDiaryPtr;
   ARFilterActionList    elseList;
   ARFilterActionList   *elseListPtr;
   char                  helpText[1024];
   char                 *helpTextPtr;
   ARNameType            name;
   ARNameType            newName;
   char                 *newNamePtr;
   unsigned int          enable;
   unsigned int         *enablePtr;
   AREscalationTmStruct escalationTm;
   AREscalationTmStruct *escalationTmPtr;
   ARNameType            owner;
   char                 *ownerPtr;
   ARQualifierStruct     query;
   ARQualifierStruct    *queryPtr;
   int                   result;
   ARNameType            schema;
   char                 *schemaPtr;
   ARStatusList          status;
   int                   i, day, date;

   /* Get Parameters */
   (void) printf("SET ESCALATION\n");
   (void) strcpy(name, GetChar("Escalation name: ", ""));
   if (GetARBoolean("Change escalation name? (F): ", FALSE))
   {
      (void) strcpy(newName, GetChar("New name (): ", ""));
      newNamePtr = newName;
   }
   else
      newNamePtr = NULL;
   if (GetARBoolean("Change escalation time ? (F): ", FALSE))
   {
      escalationTm.escalationTmType =
                                GetInt("Escalation time format (1/2) (1): ", 1);
      if (escalationTm.escalationTmType == AR_ESCALATION_TYPE_INTERVAL)
        escalationTm.u.interval = GetInt("Escalation interval (> 0) (300): ",
                                         300);
      else
      {
        escalationTm.u.date.monthday = 0;
        escalationTm.u.date.weekday = 0;
        escalationTm.u.date.hourmask = 0;
        escalationTm.u.date.minute = 0;
        day = GetInt("how many days to set in day/month format? (> 0) (1): ",1);
        i = 0;
        while (i < day)
        {
          date = GetInt("date of the day, 0 to 30 for month/day (0): ", 0);
          if (date < 0 || date > 30)
            continue;
          else
          {
            escalationTm.u.date.monthday |= AR_SETDAY(date);
            i ++;
          }
        }
        day = GetInt("how many days to set in day/week format? (> 0) (1): ", 1);
        i = 0;
        while (i < day)
        {
          date = GetInt("date of the day, 0 to 6 for sunday to saturday (0): ",
                        0);
          if (date < 0 || date > 6)
            continue;
          else
          {
            escalationTm.u.date.weekday |= AR_SETDAY(date);
            i ++;
          }
        }
        day = GetInt("how many hour marks to set in a day? (> 0) (1): ", 1);
        i = 0;
        while (i < day)
        {
          date = GetInt("hour of the day, 0 to 23 for 24 hours of a day (0): ",
                        0);
          if (date < 0 || date > 23)
            continue;
          else
          {
            escalationTm.u.date.hourmask |= AR_SETHOUR(date);
            i ++;
          }
        }
        day = GetInt("set minute mark of the hour ? (0-59) (0): ", 0);
        while (day < 0 || day > 59)
          day = GetInt("set minute mark of the hour ? (0-59) (0): ", 0);
        escalationTm.u.date.minute = day;
      }
      escalationTmPtr = &escalationTm;
   }
   else
      escalationTmPtr = NULL;
   if (GetARBoolean("Change schema? (F): ", FALSE))
   {
      (void) strcpy(schema, GetChar("Schema name (): ", ""));
      schemaPtr = schema;
   }
   else
      schemaPtr = NULL;
   if (GetARBoolean("Change enable flag? (F): ", FALSE))
   {
      enable = GetInt("Disable/Enable (0-1) (1): ", 1);
      enablePtr = &enable;
   }
   else
      enablePtr = NULL;
   if (GetARBoolean("Change query? (F): ", FALSE))
   {
      GetARQualifierStruct(&query);
      queryPtr = &query;
   }
   else
      queryPtr = NULL;
   if (GetARBoolean("Change action(s)? (F): ", FALSE))
   {
      GetARFilterActionList(TRUE, &actionList);
      actionListPtr = &actionList;
   }
   else
      actionListPtr = NULL;
   if (GetARBoolean("Change else(s)? (F): ", FALSE))
   {
      GetARFilterActionList(FALSE, &elseList);
      elseListPtr = &elseList;
   }
   else
      elseListPtr = NULL;
   if (GetARBoolean("Change help text? (F): ", FALSE))
   {
      (void) strcpy(helpText, GetChar("   Help Text (): ", ""));
      helpTextPtr = helpText;
   }
   else
      helpTextPtr = NULL;
   if (GetARBoolean("Change owner? (F): ", FALSE))
   {
      (void) strcpy(owner, GetChar("   Owner (): ", ""));
      ownerPtr = owner;
   }
   else
      ownerPtr = NULL;
   if (GetARBoolean("Add to change diary? (F): ", FALSE))
   {
      (void) strcpy(changeDiary, GetChar("   Change Diary (): ", ""));
      changeDiaryPtr = changeDiary;
   }
   else
      changeDiaryPtr = NULL;

   /* Call routine */
   result = ARSetEscalation(GetControlStructPtr(), name, newNamePtr,
                            escalationTmPtr, schemaPtr, enablePtr, queryPtr,
                            actionListPtr, elseListPtr, helpTextPtr, ownerPtr,
                            changeDiaryPtr, &status);

   /* Print results */
   DriverPrint("\n   ARSetEscalation  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   if (queryPtr != NULL)
      FreeARQualifierStruct(&query, FALSE);
   if (actionListPtr != NULL)
      FreeARFilterActionList(&actionList, FALSE);
   if (elseListPtr != NULL)
      FreeARFilterActionList(&elseList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARCreateEscalation                           */
/*                                                                           */
/*****************************************************************************/

void APIARCreateEscalation()

{
   ARFilterActionList   actionList;
   char                 changeDiary[1024];
   ARFilterActionList   elseList;
   char                 helpText[1024];
   ARNameType           name;
   unsigned int         enable;
   AREscalationTmStruct escalationTm;
   ARNameType           owner;
   ARQualifierStruct    query;
   int                  result;
   ARNameType           schema;
   ARStatusList         status;
   int                  i, day, date;

   /* Get Parameters */
   (void) printf("CREATE ESCALATION\n");
   (void) strcpy(name, GetChar("Escalation name: ", ""));
   escalationTm.escalationTmType = GetInt("Escalation time format (1/2) (1): ",
                                          1);
   if (escalationTm.escalationTmType == AR_ESCALATION_TYPE_INTERVAL)
     escalationTm.u.interval = GetInt("Escalation interval (> 0) (300): ", 300);
   else
   {
     escalationTm.u.date.monthday = 0;
     escalationTm.u.date.weekday = 0;
     escalationTm.u.date.hourmask = 0;
     escalationTm.u.date.minute = 0;
     day = GetInt("how many days to set in day/month format? (> 0) (1): ", 1);
     i = 0;
     while (i < day)
     {
       date = GetInt("date of the day, 0 to 30 for month/day (0): ", 0);
       if (date < 0 || date > 30)
         continue;
       else
       {
         escalationTm.u.date.monthday |= AR_SETDAY(date);
         i ++;
       }
     }
     day = GetInt("how many days to set in day/week format? (> 0) (1): ", 1);
     i = 0;
     while (i < day)
     {
       date = GetInt("date of the day, 0 to 6 for sunday to saturday (0): ", 0);
       if (date < 0 || date > 6)
         continue;
       else
       {
         escalationTm.u.date.weekday |= AR_SETDAY(date);
         i ++;
       }
     }
     day = GetInt("how many hour marks to set in a day? (> 0) (1): ", 1);
     i = 0;
     while (i < day)
     {
       date = GetInt("hour of the day, 0 to 23 for 24 hours of a day (0): ", 0);
       if (date < 0 || date > 23)
         continue;
       else
       {
         escalationTm.u.date.hourmask |= AR_SETHOUR(date);
         i ++;
       }
     }
     day = GetInt("set minute mark of the hour ? (0-59) (0): ", 0);
     while (day < 0 || day > 59)
       day = GetInt("set minute mark of the hour ? (0-59) (0): ", 0);
     escalationTm.u.date.minute = day;
    
   }
   (void) strcpy(schema, GetChar("Schema name: ", ""));
   enable = GetInt("Disable/Enable (0-1) (1): ", 1);
   GetARQualifierStruct(&query);
   GetARFilterActionList(TRUE, &actionList);
   GetARFilterActionList(FALSE, &elseList);
   (void) strcpy(helpText, GetChar("Help Text: ", ""));
   (void) strcpy(owner, GetChar("Owner: ", GetControlStructPtr()->user));
   (void) strcpy(changeDiary, GetChar("Change Diary: ", ""));

   /* Call routine */
   result = ARCreateEscalation(GetControlStructPtr(), name, &escalationTm,
                               schema, enable, &query, &actionList, &elseList,
                               helpText, owner, changeDiary, &status);

   /* Print results */
   DriverPrint("\n   ARCreateEscalation  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARQualifierStruct(&query, FALSE);
   FreeARFilterActionList(&actionList, FALSE);
   FreeARFilterActionList(&elseList, FALSE);
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARDeleteEscalation                           */
/*                                                                           */
/*****************************************************************************/

void APIARDeleteEscalation()

{
   ARNameType           name;
   int                  result;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("DELETE ESCALATION\n");
   (void) strcpy(name, GetChar("Escalation name: ", ""));

   /* Call routine */
   result = ARDeleteEscalation(GetControlStructPtr(), name, &status);

   /* Print results */
   DriverPrint("\n   ARDeleteEscalation  results\n");
   PrintReturnCode(result);
   PrintARStatusList(&status);

   /* Cleanup */
   FreeARStatusList(&status, FALSE);
}


/*****************************************************************************/
/*                                                                           */
/*                           APIARGetListEscalation                          */
/*                                                                           */
/*****************************************************************************/

void APIARGetListEscalation()

{
   ARTimestamp          firewithin;
   ARNameList           nameList;
   int                  result;
   ARNameType           schema;
   char                *schemaPtr;
   ARStatusList         status;

   /* Get Parameters */
   (void) printf("GETLIST ESCALATION\n");
   if (GetARBoolean("For a specific schema? (F): ", FALSE))
   {
      (void) strcpy(schema, GetChar("Schema name (): ", ""));
      schemaPtr = schema;
   }
   else
      schemaPtr = NULL;
   firewithin = GetARTimestamp("Get all to be fired within (0 means all): ", (ARTimestamp)0);

   /* Call routine */
   if (firewithin != 0)
     firewithin = firewithin + time(NULL);
   result = ARGetListEscalation(GetControlStructPtr(), schemaPtr, firewithin,
                                &nameList, &status);

   /* Print results */
   DriverPrint("\n   ARGetListEscalation  results\n");
   PrintReturnCode(result);
   if (result < AR_RETURN_ERROR)
      PrintARNameList("Escalation list: ", "   ", &nameList);
   PrintARStatusList(&status);

   /* Cleanup */
   if (result < AR_RETURN_ERROR)
      FreeARNameList(&nameList, FALSE);
   FreeARStatusList(&status, FALSE);
}
