/*  File:  daysOpen.c  */

                            /* Unix System include files */
#include    <stdio.h>
#include    <stdlib.h>
#include    <string.h>
#include    <time.h>
                            /* AR System include files */
#include    <ar.h>
#include    <arextern.h>
 

#define HOLIDAY_FILE          "holiday.txt"
                                        /* default date exception file */
#define MAX_DATE_ENTRIES      100       /* maximum number of exception dates */

                            /* structure to hold a year and day within the   */
                            /*  year to hold information about holiday dates */
typedef struct {
                  time_t time;
                  int    year;
                  int    dayInYear;
                  int    weekday;
               } YearAndDayStruct;

static time_t   currentTime;     /* current timestamp */
 
 

/*****************************************************************************/
/*                                                                           */
/*                                LoadHolidayFile                            */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Description:  Open the specified file and load the dates that are       */
/*      specified in the file.  Only dates up to a maximum are loaded and    */
/*      any invalid date format is ignored.                                  */
/*         If an error is encountered, the error string is loaded and -1 is  */
/*      returned.  Otherwise, the return is the count of the number of items */
/*      loaded from the holiday date file.                                   */
/*                                                                           */
/*   Globals Accessed:                                                       */
/*                                                                           */
/*   Modification History:                                                   */
/*       Date     Engineer  Description                                      */
/*      --------  --------  ------------                                     */
/*      05/01/93   Remedy   Procedure created                                */
/*                                                                           */
/*****************************************************************************/

int LoadHolidayFile(
char             *holidayFile, /* IN; filename of the holiday exclusion file */
YearAndDayStruct *holidays,    /* OUT; list of holiday dates      */
                               /*      array max MAX_DATE_ENTRIES */
char            **errorString  /* OUT; points to error string if error */
)

{
    char       buffer[81]; /* working buffer */
    FILE      *filePtr;    /* pointer to the open holiday date file */
    int        i;          /* working index */
    int        numDates;   /* number of dates loaded */
    struct tm  tm;         /* time structure */
 
    *errorString = NULL;

    /* Open the specified holiday dates file */
    filePtr = fopen(holidayFile, "r");
    if (filePtr == NULL)
    {    /* load an error message and return error */
        *errorString = malloc((unsigned) (80 + strlen(holidayFile)));
        if (*errorString != NULL)
            (void) sprintf(*errorString, "Cannot open holiday file %s",
                           holidayFile);
        return -1;
    }
 
    /* Populate the holiday date table from the file */
    numDates = 0;
    while (fgets(buffer, 80, filePtr) != NULL)
    {   /* trim trailing whitespace */
        i = strlen(buffer) - 1;
        while ((buffer[i] == '\n') || (buffer[i] == '\r') ||
               (buffer[i] == ' ') || (buffer[i] == '\t'))
        {
            buffer[i] = '\0';
            i--;
        }

        /* trim leading whitespace */
        i = 0;
        while ((buffer[i] == ' ') || (buffer[i] == '\t'))
            i++;

        /* if first character is a # or is an empty line, skip this line */
        if ((buffer[i] == '#') || (buffer[i] == '\0'))
            continue;

        /* convert the date that was found */
#if    defined(HP) || defined(SGI)
        /* NOTE: not all platforms support the strptime call.  If this call  */
        /*       is not supported, you will need to use another call that    */
        /*       is available or write your own routine to interpret the     */
        /*       dates in the date file.  Since it is your file, you can set */
        /*       whatever rules you choose for the format of that file.      */
        continue;  /* an alternative needed for HP; will not support holiday */
                   /* dates unless a translation routine provided here       */
	/* NOTREACHED */
#else
        if (strptime(&buffer[i], "%x", &tm) == NULL)
            continue;    /* cannot translate so skip this time */
#endif

        /* save the year and the day within the year */
        holidays[numDates].time = mktime(&tm);
        holidays[numDates].year = tm.tm_year;
        holidays[numDates].dayInYear = tm.tm_yday;
        holidays[numDates].weekday = tm.tm_wday;

        /* record another date found */
        numDates++;
    }
 
    /* close the file */
    fclose(filePtr);
 
    /* return the count of the number of dates loaded */
    return numDates;
}


/*****************************************************************************/
/*                                                                           */
/*                                PrintARStatusList                          */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Description:  Print the returned status structure to report error.      */
/*                                                                           */
/*   Globals Accessed:                                                       */
/*                                                                           */
/*   Modification History:                                                   */
/*       Date     Engineer  Description                                      */
/*      --------  --------  ------------                                     */
/*      05/01/93   Remedy   Procedure created                                */
/*                                                                           */
/*****************************************************************************/

void PrintARStatusList(value)
ARStatusList *value;          /* IN; value to print */

{
   int             i;         /* working index */
   ARStatusStruct *tempPtr;   /* working pointer */
   char           *tempPtr2;  /* working pointer */

   (void) printf("Status List : %u items\n", value->numItems);
   if (value->numItems != 0)
   {
      tempPtr = value->statusList;
      for (i = 0; i < (int) value->numItems; i++)
      {
         switch (tempPtr->messageType)
         {
            case AR_RETURN_OK         :
               tempPtr2 = "NOTE";
               break;
            case AR_RETURN_WARNING    :
               tempPtr2 = "WARNING";
               break;
            case AR_RETURN_ERROR      :
               tempPtr2 = "ERROR";
               break;
            default                   :
               tempPtr2 = "<unknown type>";
               break;
         }
         (void) printf(
              "Status Struct :\n   Message type : %s\n   Message number : %d\n",
              tempPtr2, tempPtr->messageNum);
         if (tempPtr->messageText == NULL)
            (void) printf("   Message: \n");
         else
            (void) printf("   Message: %s\n", tempPtr->messageText);
         if (tempPtr->appendedText != NULL && tempPtr->appendedText[0])
            (void) printf("   Appended: %s\n", tempPtr->appendedText);
         tempPtr++;
      }
   }
}


/*****************************************************************************/
/*                                                                           */
/*                                 ComputeDaysOpen                           */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Description:  Given a start and end date, compute the number of days a  */
/*      problem has been opened.  This routine will exclude weekends and     */
/*      and dates that fall on a holiday.                                    */
/*         The number of days is the total number of days the problem has    */
/*      been open at the end of the day.  If a problem is closed on the same */
/*      day as it was entered, the return is 0 (open at the end of the day   */
/*      for 0 full days).  If a problem is still open at the end of the first*/
/*      day, the return is 1 (open at the end of 1 day).  If a problem is    */
/*      closed on a weekend or a holiday, it is the same as if the problem   */
/*      was closed on the next working day (was open at the end of the       */
/*      previous working day).                                               */
/*         If the start date is later than the end date, this routine returns*/
/*      a -1 as an indicator of a problem.                                   */
/*         NOTE: This algorithm works with midnight normalized time which    */
/*      means that all time values are ignored and only days considered.  So,*/
/*      the program must be run after midnight to include the previous day   */
/*      as a day for any problem not closed yet.                             */
/*         REMEMBER:  The rules discussed here are just examples of rules    */
/*      that could be used.  You may want to count the number of hours the   */
/*      problem was open instead of the number of days.  You may want to     */
/*      count problems closed on holidays more favorably than assuming the   */
/*      close on the next working day.  You may want to include weekends in  */
/*      your count so the weekend elimination calculation is omitted.  The   */
/*      key concept is that you can change the rules to be whatever you want */
/*      without affecting the overall approach or structure of the program.  */
/*      (You may even want to support multiple of these styles with the      */
/*      configuration options to indicate the type of calculation done.)     */
/*                                                                           */
/*   Globals Accessed:                                                       */
/*      currentTime       OUT     Set to today's timestamp                   */
/*                                                                           */
/*   Modification History:                                                   */
/*       Date     Engineer  Description                                      */
/*      --------  --------  ------------                                     */
/*      05/01/93   Remedy   Procedure created                                */
/*                                                                           */
/*****************************************************************************/

int ComputeDaysOpen(
long              startDate,   /* IN; starting date to compute days open from */
long              endDate,     /* IN; ending date to compute days open to */
int               numHolidays, /* IN; number of holiday dates */
YearAndDayStruct *holidays     /* IN; list of holiday dates */
)

{
    int        endDay;               /* ending day of week */
    int        holidayDays;          /* number of holiday days was open */
    int        i;                    /* working index */
    long       normalizedEndDate;    /* end date normalized to midnight */
    long       normalizedStartDate;  /* start date normalized to midnight */
    int        startDay;             /* starting day of week */
    struct tm *tm;                   /* time structure */
    int        totalDays;            /* total days problem was open */
    int        weekendDays;          /* number of weekend days was open */

    /* Check that startDate is before endDate.  If not, return -1 */
    if (startDate > endDate)
        return -1;

    /* Normalize the dates to midnight.  Normalization occurs by subtracting */
    /* the total number of seconds represented by hours, minutes, seconds    */
    /* from the start time.                                                  */
    tm = localtime(&startDate);
    normalizedStartDate = startDate - (tm->tm_hour * 60 * 60) -
                          (tm->tm_min * 60) - tm->tm_sec;
    startDay = tm->tm_wday;
    tm = localtime(&endDate);
    normalizedEndDate = endDate - (tm->tm_hour * 60 * 60) -
                        (tm->tm_min * 60) - tm->tm_sec;
    endDay = tm->tm_wday;

    /* If the normalized dates are the same, return 0 days outstanding */
    if (normalizedStartDate == normalizedEndDate)
        return 0;

    /* The difference of the normalized dates gives the total number of days */
    /* a call was open (the number of days it was open at the end of the day)*/
    totalDays = (endDate - startDate) / 60 / 60 / 24;

    /* Now, need to remove weekends.  If take the total number of days and */
    /* divide by 7, get the number of complete weeks, each of which has a  */
    /* 2 day weekend.  The remainder of a division by 7 are the extra days.*/
    /* A check of them can be used to determine how many weekend days are  */
    /* included in the partial week.                                       */
    weekendDays = (totalDays / 7) * 2;  /* complete weekends */

    /* Add weekend for partial week */
    while (startDay != endDay)
    {
       if ((startDay == 0) || (startDay == 6))
           weekendDays++;               /* another weekend day */
       if (startDay == 6)
           startDay = 0;
       else
           startDay++;
    }

    /* Compute total number of days of holidays in the interval.  Remember, */
    /* need to check each holiday and if is a weekend, exclude it.          */
    holidayDays = 0;
    for (i = 0; i < numHolidays; i++)
       if ((holidays[i].weekday != 0) && (holidays[i].weekday != 6) &&
           (holidays[i].time >= startDate) && (holidays[i].time <= endDate))
           holidayDays++;              /* another holiday day */

    /* finally, return the total days less weekends and non-weekend holidays */
    return totalDays - weekendDays - holidayDays;
}


/*****************************************************************************/
/*                                                                           */
/*                               CalculateOneDayOpen                         */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Description:  Compute the daysOpen result for a single, specified entry.*/
/*      Build the structures and call the routine to perform the calculation */
/*      only on the entry specified.                                         */
/*                                                                           */
/*   Globals Accessed:                                                       */
/*      currentTime       OUT     Set to today's timestamp                   */
/*                                                                           */
/*   Modification History:                                                   */
/*       Date     Engineer  Description                                      */
/*      --------  --------  ------------                                     */
/*      05/01/93   Remedy   Procedure created                                */
/*                                                                           */
/*****************************************************************************/

int CalculateOneDayOpen(
ARControlStruct  *control,     /* IN; control record for the operation */
ARNameType        schema,      /* IN; schema to perform operation on */
ARInternalId      startFieldId,/* IN; id of field holding start date */
ARInternalId      endFieldId,  /* IN; id of field holding end date */
ARInternalId      daysOpenId,  /* IN; id of field holding days open result */
int               numHolidays, /* IN; number of holiday dates */
YearAndDayStruct *holidays,    /* IN; list of holiday dates */
AREntryIdList    *entryId      /* IN; id of entry to calculate value for */
)

{
    int                daysOpen;   /* number of days open so far */
    int                daysOpen2;  /* number of days open so far */
    long               endDate;    /* end timestamp */
    ARFieldValueList   fieldList;  /* field/value list of retrieved data */
    ARFieldValueStruct fieldStruct;/* buffer to hold field/value information */
    int                i;          /* working index */
    ARInternalIdList   idList;     /* list of ids for fields to retrieve */
    ARInternalId       ids[3];     /* ids of field values to retrieve */
    int                returnCode; /* return code from subroutine*/
    long               startDate;  /* start timestamp */
    ARStatusList       status;     /* status of AR System op */

    /* Load the id list.  Since it is known that we want 3 values, I can */
    /* build the list in local variables without mallocing space.        */
    idList.numItems = 3;
    idList.internalIdList = ids;
    ids[0] = startFieldId;
    ids[1] = endFieldId;
    ids[2] = daysOpenId;    /* get current value to save update if unchanged! */
    /* since this is the same list needed throughout the operation if all    */
    /* entries are being updated, it would be more efficient to load it once */
    /* outside this routine and pass the data in.  It is not terrible since  */
    /* it is a set of 5 local assigns so we didn't bother in this example.   */

    /* retrieve the entries from the specified record */
    returnCode = ARGetEntry(control, schema, entryId, &idList, &fieldList,
                            &status);
    if (returnCode >= AR_RETURN_ERROR)
    {
        PrintARStatusList(&status);
        return returnCode;
    }

    /* REMEMBER to free the space associated with the returned status */
    /* structure as may include warnings and notes we are ignoring    */
    FreeARStatusList(&status, FALSE);

    /* OK return, retrieve the data values.  They should be in the same   */
    /* order as the idList, but just in case the idList ever changes, we  */
    /* will check the values.  This also helps in case of warning where a */
    /* value was not available.                                           */
    startDate = 0;
    endDate = 0;
    daysOpen = -1;
    for (i = 0; i < fieldList.numItems; i++)
    {
        if (fieldList.fieldValueList[i].fieldId == startFieldId)
        {
            if (fieldList.fieldValueList[i].value.dataType == AR_DATA_TYPE_TIME)
                startDate = fieldList.fieldValueList[i].value.u.timeVal;
            /*else   must be NULL so is no value */
        }
        else if (fieldList.fieldValueList[i].fieldId == endFieldId)
        {
            if (fieldList.fieldValueList[i].value.dataType == AR_DATA_TYPE_TIME)
                endDate = fieldList.fieldValueList[i].value.u.timeVal;
            /*else   must be NULL so is no value */
        }
        else if (fieldList.fieldValueList[i].fieldId == daysOpenId)
        {
            if (fieldList.fieldValueList[i].value.dataType ==
                AR_DATA_TYPE_INTEGER)
                daysOpen = fieldList.fieldValueList[i].value.u.intVal;
            /*else   must be NULL so is no value */
        }
    }

    /* REMEMBER to free the space associated with the returned fieldList */
    FreeARFieldValueList(&fieldList, FALSE);

    /* if no start date, cannot perform operation */
    if (startDate == 0)
    {
       (void) printf("No start date value for entry %s\n",
                     entryId->entryIdList[0]);
       return AR_RETURN_ERROR;
    }

    /* Compute the number of days the problem was/has been open.  If there */
    /* is a close date, compute the difference between start and end.  If  */
    /* not, compute the difference between the start and today.            */

    /* if no end date, use todays date */
    if (endDate == 0)
        endDate = currentTime;

    /* compute the resulting number of days open */
    daysOpen2 = ComputeDaysOpen(startDate, endDate, numHolidays, holidays);

    /* Check to see if the number of days open calculated is the same as the */
    /* value that is already held in the database.  This check is useful to  */
    /* eliminate updates where the value is already correct.                 */
    if (daysOpen != daysOpen2)
    {   /* update the daysOpen value */
        fieldList.numItems = 1;
        fieldList.fieldValueList = &fieldStruct;

        fieldStruct.fieldId = daysOpenId;
        fieldStruct.value.dataType = AR_DATA_TYPE_INTEGER;
        fieldStruct.value.u.intVal = daysOpen2; 

        /* Perform the update. Note: we pass 0 as the get time so that the    */
        /* update is performed.  We are only updating the daysOpen field      */
        /* which is owned by this program so there should be no conflict with */
        /* any updating done by users.                                        */
        returnCode = ARSetEntry(control, schema, entryId, &fieldList, 0,
                                AR_JOIN_SETOPTION_NONE, &status);
        if (returnCode >= AR_RETURN_ERROR)
        {
            PrintARStatusList(&status);
            return returnCode;
        }

        /* REMEMBER to free the space associated with the returned status */
        /* structure as may include warnings and notes we are ignoring    */
        FreeARStatusList(&status, FALSE);
    }
    /*else    already the same so no action is needed */

    /* return a successful operation */
    return AR_RETURN_OK;
}


/*****************************************************************************/
/*                                                                           */
/*                                CalculateAllDaysOpen                       */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Description:  Compute the daysOpen result for all entries in the schema */
/*      that are not closed (no close date) or that are closed for less than */
/*      one week.  Get a list of all the entries that match this condition   */
/*      and build the structures and call the routine to perform the         */
/*      calculation for each entry found.                                    */
/*                                                                           */
/*   Globals Accessed:                                                       */
/*      currentTime       OUT     Set to today's timestamp                   */
/*                                                                           */
/*   Modification History:                                                   */
/*       Date     Engineer  Description                                      */
/*      --------  --------  ------------                                     */
/*      05/01/93   Remedy   Procedure created                                */
/*                                                                           */
/*****************************************************************************/

int CalculateAllDaysOpen(
ARControlStruct  *control,     /* IN; control record for the operation */
ARNameType        schema,      /* IN; schema to perform operation on */
ARInternalId      startFieldId,/* IN; id of field holding start date */
ARInternalId      endFieldId,  /* IN; id of field holding end date */
ARInternalId      daysOpenId,  /* IN; id of field holding days open result */
int               numHolidays, /* IN; number of holiday dates */
YearAndDayStruct *holidays     /* IN; list of holiday dates */
)

{
    ARArithOpStruct   arithOp;    /* arithmetic operation for subtract */
    AREntryListList   entryList;  /* list of entries to update */
    int               i;          /* working index */
    ARQualifierStruct qualifier;  /* root structure of qualification */
    ARQualifierStruct qualifier2; /* qualification to check for NULL end date */
    ARQualifierStruct qualifier3; /* qualification to check for old  end date */
    ARRelOpStruct     relOp1;     /* rel op for the null check */
    ARRelOpStruct     relOp2;     /* rel op for the today check */
    int               returnCode; /* return code from subroutine*/
    ARStatusList      status;     /* status of AR System op */
 
    /* Load the qualifier struct to constrain the data loaded to meet the */
    /* the restrictions that are discussed above.                         */
    qualifier.operation = AR_COND_OP_OR;
    qualifier.u.andor.operandLeft = &qualifier2;
    qualifier.u.andor.operandRight = &qualifier3;

    /* build piece that is    'end date' = NULL   */
    qualifier2.operation = AR_COND_OP_REL_OP;
    qualifier2.u.relOp = &relOp1;
    relOp1.operation = AR_REL_OP_EQUAL;
    relOp1.operandLeft.tag = AR_FIELD;
    relOp1.operandLeft.u.fieldId = endFieldId;
    relOp1.operandRight.tag = AR_VALUE;
    relOp1.operandRight.u.value.dataType = AR_DATA_TYPE_NULL;

    /* build piece that is    'end date' > current time - one week  */
    qualifier3.operation = AR_COND_OP_REL_OP;
    qualifier3.u.relOp = &relOp2;
    relOp2.operation = AR_REL_OP_GREATER;
    relOp2.operandLeft.tag = AR_FIELD;
    relOp2.operandLeft.u.fieldId = endFieldId;
    relOp2.operandRight.tag = AR_ARITHMETIC;
    relOp2.operandRight.u.arithOp = &arithOp;
    arithOp.operation = AR_ARITH_OP_SUBTRACT;
    arithOp.operandLeft.tag = AR_VALUE;
    arithOp.operandLeft.u.value.dataType = AR_DATA_TYPE_TIME;
    arithOp.operandLeft.u.value.u.timeVal = currentTime;
    arithOp.operandRight.tag = AR_VALUE;
    arithOp.operandRight.u.value.dataType = AR_DATA_TYPE_INTEGER;
    arithOp.operandRight.u.value.u.intVal = 60 * 60 * 24 * 7;  /* one week */

    /* Retrieve all the records in the schema */
    returnCode =  ARGetListEntry(control, schema, &qualifier, NULL, NULL,
                                 AR_NO_MAX_LIST_RETRIEVE, &entryList,
                                 (unsigned int *) NULL, &status);
    if (returnCode >= AR_RETURN_ERROR)
    {
        PrintARStatusList(&status);
        return returnCode;
    }

    /* REMEMBER to free the space associated with the returned status */
    /* structure as may include warnings and notes we are ignoring    */
    FreeARStatusList(&status, FALSE);

    /* Have a list of items, call to update each of them */
    for (i = 0; i < entryList.numItems; i++) 
    {
        returnCode = CalculateOneDayOpen(control, schema, startFieldId,
                                         endFieldId, daysOpenId, numHolidays,
                                         holidays,
                                         &entryList.entryList[i].entryId);
        if (returnCode >= AR_RETURN_ERROR)
        {
            FreeAREntryListList(&entryList, FALSE);
            return returnCode;
        }
    }

    /* REMEMBER to free the space associated with the returned fieldList */
    FreeAREntryListList(&entryList, FALSE);

    /* return a successful operation */
    return AR_RETURN_OK;
}


/*****************************************************************************/
/*                                                                           */
/*                                 main block                                */
/*                                                                           */
/*****************************************************************************/
/*                                                                           */
/*   Description:  Program to take schema, server, start time field, and     */
/*      end time field as input, compute the elapsed number of days, and     */
/*      update the result field specified with the number of working days    */
/*      the problem was open.                                                */
/*         NOTE:  If the problem is still open, this will compute the total  */
/*      number of working days this problem has been open so far.            */
/*         This example assumes a pair of fields holding the start date and  */
/*      close date.  The start date is likely the Create-date core field.    */
/*      The end date is a field that you have added.  You could also key     */
/*      from the dates on the various states within the Status-History field */
/*      instead of the extra date fields.  The only change would be to       */
/*      specify the start state and the end state then read the values from  */
/*      the Status-History field.                                            */
/*                                                                           */
/*   Globals Accessed:                                                       */
/*      currentTime       OUT     Set to today's timestamp                   */
/*                                                                           */
/*   Modification History:                                                   */
/*       Date     Engineer  Description                                      */
/*      --------  --------  ------------                                     */
/*      05/01/93   Remedy   Procedure created                                */
/*                                                                           */
/*****************************************************************************/

int main(
int   argc,            /* IN; number of command line arguments */
char *argv[]           /* IN; pointers to command line arguments */
)

{
    ARControlStruct  control;                   /* control record for API */
    ARInternalId     daysOpenId;                /* days open field id */
    ARInternalId     endDateId;                 /* end date field id */
    AREntryIdType    entryId;                   /* holds entryid if specified */
    AREntryIdList    entryIdList;               /* ID put into structure */
    char            *errorString;               /* string if errors */
    YearAndDayStruct holidays[MAX_DATE_ENTRIES];/* holiday table */
    char            *holidayFile;               /* pointer to holiday filename*/
    int              numberOfHolidays;          /* number holiday dates */
    int              returnCode;                /* return code from subroutine*/
    ARNameType       schema;                    /* AR schema name */
    ARInternalId     startDateId;               /* start date field id */
    ARStatusList     status;                    /* status of AR System op */
 
    /* For this example, we use positional arguments, all but one of    */
    /* which must be supplied.  The arguments expected are as follows:  */
    /*      schema         name of the schema that is being updated     */
    /*      server         name of the server the schema is located on  */
    /*      entryId        id of specific entry (may be empty, "")      */
    /*      start date id  id of the field holding the start date       */
    /*      end date id    id of the field holding the end date         */
    /*      days open id   id of the field to hold the number days open */
    /*      holiday file   filename of holiday file (if not supplied    */
    /*                        this field will default)                  */
    if ((argc == 7) || (argc == 8))
    {
        (void) strncpy(schema, argv[1], AR_MAX_NAME_SIZE);
        schema[AR_MAX_NAME_SIZE] = '\0';
        (void) strncpy(control.server, argv[2], AR_MAX_NAME_SIZE);
        control.server[AR_MAX_NAME_SIZE] = '\0';
        (void) strncpy(entryId, argv[3], AR_MAX_ENTRYID_SIZE);
        entryId[AR_MAX_ENTRYID_SIZE] = '\0';
        startDateId = atol(argv[4]);
        endDateId = atol(argv[5]);
        daysOpenId = atol(argv[6]);

        if (argc == 8)
            holidayFile = argv[7];
        else
            holidayFile = HOLIDAY_FILE;
    }
    else
    {
        (void) printf("Usage: %s  %s  %s\n", argv[0],
                      "schema-name  server-name  entryId  start-date-field-id",
                      "end-date-field-id  days-open-field-id  [holiday-file]");
        exit(1);
    }
    /* This example uses a positional list of parameters.  You could also */
    /* hard-code values if you are referencing a specific environment or  */
    /* use keyword parameters, or use a configuration file, or ....       */

    /* Load holiday array.  This loads all the excluded holidays for this  */
    /* site.  The resulting numberOfHolidays count will contain the number */
    /* of holidays identified.                                             */
    numberOfHolidays = LoadHolidayFile(holidayFile, holidays, &errorString);
    if (numberOfHolidays < 0)
        if ((numberOfHolidays == -1) && (holidayFile == HOLIDAY_FILE))
            numberOfHolidays = 0;   /* no file specified and none found */
        else
        {
            if (errorString != NULL)
                (void) printf("%s\n", errorString);
            exit(2);
        }

    /* Get the current timestamp */
    currentTime = time((time_t *) NULL);

    /* Establish the control structure */
    control.cacheId = 0;              /* initialize cache id to 0 */
    strcpy(control.user, "Demo");     /* use the Demo user (could pass in or */
    strcpy(control.password, "");     /*  load from a config file)           */
    strcpy(control.language, "");     /* use default language ("C") */
    control.sessionId = 0;            /* initialize session id to 0 */
    /*strcpy(control.server, ...);*/  /* server loaded above */

    /* Initialize Remedy session.  This call establishes the environment for */
    /* interaction with the AR System API.  It must be the first AR API call */
    /* made in every application.                                            */
    if (ARInitialization(&control, &status) >= AR_RETURN_ERROR)
    {
        PrintARStatusList(&status);
        exit(3);
    }
    FreeARStatusList(&status, FALSE);

    /* Check the entryId parameter.  If it is an empty string, no specific   */
    /* entryId was specified so call a routine to retrieve a list of all the */
    /* entries in the system that are open or have been closed less than a   */
    /* week and compute the daysOpen field.  If the string has a value,      */
    /* call to compute a daysOpen value for just that one entry.             */
    if (entryId[0] != '\0')
    {
        entryIdList.numItems = 1;
        entryIdList.entryIdList = &entryId;
        returnCode = CalculateOneDayOpen(&control, schema, startDateId,
                                         endDateId, daysOpenId,
                                         numberOfHolidays, holidays,
                                         &entryIdList);
    }
    else
        returnCode = CalculateAllDaysOpen(&control, schema, startDateId,
                                          endDateId, daysOpenId,
                                          numberOfHolidays, holidays);

    /* check error */
    if (returnCode >= AR_RETURN_ERROR)
    {
        /* error already printed in Calculate routines */
        (void) ARTermination(&control, &status);  /* remember to terminate */
                                                  /* on error!             */
        exit(4);
    }

    /* Terminate the session with Remedy.  This call cleans up the          */
    /* environment from interaction with the AR System API.  It must be the */
    /* last AR API call made in every application.                          */
    (void) ARTermination(&control, &status);

    exit(0);
    /* NOTREACHED */
    return 0;
}
